﻿using System;
using System.Windows.Forms;
using XdsObjects;
using XdsObjects.Enums;

namespace XDS_Document_Source_Consumer__Advanced_Version_
{
    public partial class DocConsumer : Form
    {
        public DocConsumer()
        {
            InitializeComponent();
            ActionLists.SelectedIndex = 0;
            QueryReturnTypeBox.SelectedIndex = 0;
            Setup();
        }

        #region "Variable"
        private XdsDomain xds;
        // The type of the registry stored query to perform
        private QueryType queryType;
        // The type of the query response
        private QueryReturnType queryReturnType;
        // The type of the association when we do GetRelatedDocument query
        private AssociationType assocType;
        // The gateway address, when we do Cross Community Access
        private const string XCAInitiatingGatewayForQuery = "http://localhost:8020/InitQuery"; //"http://ihexds.nist.gov:12090/tf6/services/ig" - this is NIST public gateway for query and retrieve
        private const string XCAInitiatingGatewayForRetrieve = "http://localhost:8021/InitRetrieve"; //"http://ihexds.nist.gov:12090/tf6/services/ig" - this is NIST public gateway for query and retrieve
        #endregion

        #region "Setup Routine"
        /// <summary>
        /// The setup routine
        /// </summary>
        private void Setup()
        {
            // Start logging
            XdsGlobal.LogToFile(@"C:\Xds Log Files", LogLevel.FullXmlIn | LogLevel.FullXmlOut, 60);

            xds = new XdsDomain();
            // The repository address - where to send the document to
            // We set up the registry later
            xds.RepositoryAddresses.Add("1.19.6.24.109.42.1.5", new WebServiceEndpoint(RepositoryAddrText.Text));

            // The ID root of the current doamin we are in
            xds.XdsDomainRoot = "1.19.6.24.109.42.1.3";
        }
        #endregion

        #region "UI Handler and Helper Routine"

        /// <summary>
        /// Click button to perform Registry Stored Query
        /// </summary>        
        private void QueryBTN_Click(object sender, EventArgs e)
        {
            InternalQuery();
        }

        /// <summary>
        /// Routine to Display the Query Response
        /// </summary>        
        private void DisplayQueryResponse(XdsQueryResponse resp, QueryType queryType, string PatientID)
        {
            DocsTree.Nodes.Clear();

            TreeNode respNode = new TreeNode("Query Response for " + queryType.ToString());

            TreeNode errorNode = respNode.Nodes.Add("Errors (" + resp.Errors.ErrorList.Count.ToString() + ")");
            foreach (XdsErrorItem err in resp.Errors.ErrorList)
            {
                errorNode.Nodes.Add("Error:" + err.ToString());//"Severity : " + err.Severity + " codeContext: " + err.CodeContext);
            }

            TreeNode statNode = respNode.Nodes.Add("Status :" + resp.Status.ToString());

            TreeNode docNode = respNode.Nodes.Add("Returned Documents (" + resp.ReturnedDocuments.Count.ToString() + ")");
            foreach (XdsDocument doc in resp.ReturnedDocuments)
            {
                TreeNode node = docNode.Nodes.Add("MimeType: " + doc.MimeType + "   UID: " + doc.UniqueID);
                node.Tag = doc;
            }

            TreeNode folNode = respNode.Nodes.Add("Returned Folders (" + resp.ReturnedFolders.Count.ToString() + ")");
            foreach (XdsFolder fol in resp.ReturnedFolders)
            {
                folNode.Nodes.Add("FolderName: " + fol.Title + "   UID: " + fol.UniqueID);
            }

            TreeNode submNode = respNode.Nodes.Add("Returned SubmissionSets (" + resp.ReturnedSubmissionSets.Count.ToString() + ")");
            foreach (XdsSubmissionSet sub in resp.ReturnedSubmissionSets)
            {
                submNode.Nodes.Add("UID: " + sub.UniqueID);
            }

            TreeNode assocNode = respNode.Nodes.Add("Returned Associations(" + resp.ReturnedAssociations.Count.ToString() + ")");
            foreach (XdsAssociation sub in resp.ReturnedAssociations)
            {
                assocNode.Nodes.Add("UID: " + sub.UUID);
            }

            TreeNode objRefNode = respNode.Nodes.Add("Returned ReturnedObjectRefs (" + resp.ReturnedObjectRefs.Count.ToString() + ")");
            foreach (XdsObjectRef ObjRef in resp.ReturnedObjectRefs)
            {
                TreeNode n = new TreeNode(ObjRef.UUID);
                n.Tag = ObjRef;
                objRefNode.Nodes.Add(n);
            }

            DocsTree.Nodes.Add(respNode);
            DocsTree.ExpandAll();
        }

        /// <summary>
        /// Select different QueryReturnType from the dropdown list
        /// </summary>        
        private void QueryReturnTypeBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            queryReturnType = (QueryReturnType)Enum.Parse(typeof(QueryReturnType), QueryReturnTypeBox.SelectedItem.ToString());
        }

        /// <summary>
        /// Double click to retrieve the selected document
        /// </summary>        
        private void DocsTree_DoubleClick(object sender, EventArgs e)
        {
            if (DocsTree.SelectedNode.Tag is XdsDocument)
            {
                Retrieve((XdsDocument)DocsTree.SelectedNode.Tag);
            }
        }

        /// <summary>
        /// Select different AssociationType from the dropdown list
        /// </summary>        
        private void AssocType_SelectedIndexChanged(object sender, EventArgs e)
        {
            assocType = (AssociationType)Enum.Parse(typeof(AssociationType), AssocType.SelectedItem.ToString());
        }

        /// <summary>
        /// Select different type of the Registry Stored Query to perform from the dropdown list
        /// </summary>        
        private void ActionLists_SelectedIndexChanged(object sender, EventArgs e)
        {
            switch (ActionLists.SelectedItem.ToString())
            {
                case "[11936] Find Documents Stored Query":
                    queryType = QueryType.FindDocuments;
                    Tabs.SelectedIndex = 0;
                    break;
                case "[11937] Find SubmissionSets Stored Query":
                    queryType = QueryType.FindSubmissionSets;
                    Tabs.SelectedIndex = 0;
                    break;
                case "[11938] Find Folders Stored Query":
                    queryType = QueryType.FindFolders;
                    Tabs.SelectedIndex = 10;
                    break;
                case "[11939] GetAll Stored Query":
                    queryType = QueryType.GetAll;
                    Tabs.SelectedIndex = 0;
                    break;
                case "[11940] GetDocuments Stored Query":
                    queryType = QueryType.GetDocuments;
                    Tabs.SelectedIndex = 1;
                    break;
                case "[11941] GetFolders Stored Query":
                    queryType = QueryType.GetFolders;
                    Tabs.SelectedIndex = 2;
                    break;
                case "[11942] GetAssociations Stored Query":
                    queryType = QueryType.GetAssociations;
                    Tabs.SelectedIndex = 3;
                    break;
                case "[11905] GetSubmissionSets Stored Query":
                    queryType = QueryType.GetSubmissionSets;
                    Tabs.SelectedIndex = 4;
                    break;
                case "[11943] GetDocumentsAndAssociations Stored Query":
                    queryType = QueryType.GetDocumentsAndAssociations;
                    Tabs.SelectedIndex = 5;
                    break;
                case "[11945] GetSubmissionSetAndContents Stored Query":
                    queryType = QueryType.GetSubmissionSetAndContents;
                    Tabs.SelectedIndex = 6;
                    break;
                case "[11946] GetFolderAndContents Stored Query":
                    queryType = QueryType.GetFolderAndContents;
                    Tabs.SelectedIndex = 7;
                    break;
                case "[11947] GetFoldersForDocument Stored Query":
                    queryType = QueryType.GetFoldersForDocument;
                    Tabs.SelectedIndex = 8;
                    break;
                case "[11948] GetRelatedDocuments Stored Query":
                    queryType = QueryType.GetRelatedDocuments;
                    Tabs.SelectedIndex = 9;
                    break;
            }
        }

        #endregion

        #region "Query and Retrieve"
        /// <summary>
        /// Query
        /// </summary>
        private void InternalQuery()
        {
            if (queryType == QueryType.None)
            {
                MessageBox.Show("Selected Query Type First");
                return;
            }

            if (queryReturnType == QueryReturnType.None)
            {
                MessageBox.Show("Selected Query Return Type First");
                return;
            }

            DocsTree.Nodes.Clear();

            XdsQueryRequest q = new XdsQueryRequest();
            q.PatientId = PIDText.Text;
            q.QueryReturnType = queryReturnType;
            q.QueryType = queryType;
            q.DocumentEntryStatus = AvailabilityStatusCode.All;

            switch (q.QueryType)
            {
                case QueryType.GetDocuments: // DocumentEntryEntryUUID or DocumentEntryUniqueId MUST be provided for GetDocuments
                    if ((string.IsNullOrEmpty(GetDocs_DocumentEntryEntryUUIDText.Text) && string.IsNullOrEmpty(GetDocs_DocumentEntryUniqueIdText.Text))
                        || (!string.IsNullOrEmpty(GetDocs_DocumentEntryEntryUUIDText.Text) && (!string.IsNullOrEmpty(GetDocs_DocumentEntryUniqueIdText.Text))))
                        throw new Exception("Must have one and only one Value");

                    if (!string.IsNullOrEmpty(GetDocs_DocumentEntryEntryUUIDText.Text))
                    {
                        if (GetDocs_DocumentEntryEntryUUIDText.Text.Contains("||"))
                        {
                            String[] uuids = GetDocs_DocumentEntryEntryUUIDText.Text.Split(new String[] { "||" }, System.StringSplitOptions.None);
                            foreach (string uuid in uuids)
                            {
                                q.DocumentEntryEntryUUID.Add(uuid);
                            }
                        }
                        else
                            q.DocumentEntryEntryUUID.Add(GetDocs_DocumentEntryEntryUUIDText.Text);
                    }
                    if (!string.IsNullOrEmpty(GetDocs_DocumentEntryUniqueIdText.Text))
                        q.DocumentEntryUniqueId.Add(GetDocs_DocumentEntryUniqueIdText.Text);

                    break;

                case QueryType.GetFolders: // FolderEntryUUID or FolderUniqueId MUST be provided for GetFolders
                    if ((string.IsNullOrEmpty(GetFolders_FolderEntryUUIDText.Text) && string.IsNullOrEmpty(GetFolders_FolderUniqueIdText.Text))
                       || (!string.IsNullOrEmpty(GetFolders_FolderEntryUUIDText.Text) && (!string.IsNullOrEmpty(GetFolders_FolderUniqueIdText.Text))))
                        throw new Exception("Must have one and only one Value");

                    if (!string.IsNullOrEmpty(GetFolders_FolderEntryUUIDText.Text))
                        q.FolderEntryUUID.Add(GetFolders_FolderEntryUUIDText.Text);

                    if (!string.IsNullOrEmpty(GetFolders_FolderUniqueIdText.Text))
                        q.FolderUniqueId.Add(GetFolders_FolderUniqueIdText.Text);
                    break;

                case QueryType.GetAssociations: // AssociationType MUST be provided for GetAssociations
                    if (string.IsNullOrEmpty(GetAsso_UUIDText.Text))
                        throw new Exception("Association UUID Must be set");
                    else
                    {
                        if (GetAsso_UUIDText.Text.Contains("||"))
                        {
                            string[] uuids = GetAsso_UUIDText.Text.Split(new string[] { "||" }, StringSplitOptions.None);
                            foreach (string uuid in uuids)
                            {
                                q.AssociationUUID.Add(uuid);
                            }
                        }
                        else
                        {
                            q.AssociationUUID.Add(GetAsso_UUIDText.Text);
                        }
                    }
                    break;

                case QueryType.GetSubmissionSets: // DocumentEntryEntryUUID and FolderEntryUUID cannot be both null for GetSubmissionSets
                    if (string.IsNullOrEmpty(GetSubmissionSets_DocumentEntryEntryUUIDText.Text) && string.IsNullOrEmpty(GetSubmissionSets_FolderEntryUUID.Text))
                        throw new Exception("DocumentEntryEntryUUID and FolderEntryEntryUUID Cannot be both Null");

                    //TODO:
                    if (!string.IsNullOrEmpty(GetSubmissionSets_DocumentEntryEntryUUIDText.Text))
                        //q.DocumentEntryEntryUUID.Add(GetSubmissionSets_DocumentEntryEntryUUIDText.Text);
                        q.AssociationUUID.Add(GetSubmissionSets_DocumentEntryEntryUUIDText.Text);

                    if (!string.IsNullOrEmpty(GetSubmissionSets_FolderEntryUUID.Text))
                        //q.FolderEntryUUID.Add(GetSubmissionSets_FolderEntryUUID.Text);
                        q.AssociationUUID.Add(GetSubmissionSets_FolderEntryUUID.Text);

                    break;

                case QueryType.GetDocumentsAndAssociations: // DocumentEntryEntryUUID or DocumentEntryUniqueId MUST be provide for GetDocumentsAndAssociations
                    if ((string.IsNullOrEmpty(GetDocAsso_DocumentEntryEntryUUIDtext.Text) && string.IsNullOrEmpty(GetDocAsso_DocumentEntryUniqueIdText.Text))
                       || (!string.IsNullOrEmpty(GetDocAsso_DocumentEntryEntryUUIDtext.Text) && (!string.IsNullOrEmpty(GetDocAsso_DocumentEntryUniqueIdText.Text))))
                        throw new Exception("Must have one and only one Value");

                    if (!string.IsNullOrEmpty(GetDocAsso_DocumentEntryEntryUUIDtext.Text))
                        q.DocumentEntryEntryUUID.Add(GetDocAsso_DocumentEntryEntryUUIDtext.Text);

                    if (!string.IsNullOrEmpty(GetDocAsso_DocumentEntryUniqueIdText.Text))
                        q.DocumentEntryUniqueId.Add(GetDocAsso_DocumentEntryUniqueIdText.Text);
                    break;

                case QueryType.GetSubmissionSetAndContents: // SubmissionSetEntryUUID or SubmissionSetUniqueId MUST be provide for GetSubmissionSetAndContents
                    if ((string.IsNullOrEmpty(GetSubSetAndCont_SubmissionSetEntryUUIDText.Text) && string.IsNullOrEmpty(GetSubSetAndCont_SubmissionSetUniqueIdText.Text))
                      || (!string.IsNullOrEmpty(GetSubSetAndCont_SubmissionSetEntryUUIDText.Text) && (!string.IsNullOrEmpty(GetSubSetAndCont_SubmissionSetUniqueIdText.Text))))
                        throw new Exception("Must have one and only one Value");

                    if (!string.IsNullOrEmpty(GetSubSetAndCont_SubmissionSetEntryUUIDText.Text))
                        q.SubmissionSetEntryUUID = new XdsObjectRef(GetSubSetAndCont_SubmissionSetEntryUUIDText.Text);

                    if (!string.IsNullOrEmpty(GetSubSetAndCont_SubmissionSetUniqueIdText.Text))
                        q.SubmissionSetUniqueId = new XdsObjectRef(GetSubSetAndCont_SubmissionSetUniqueIdText.Text);

                    break;

                case QueryType.GetFolderAndContents: //FolderEntryUUID or FolderUniqueID MUST be provide for GetFolderAndContents
                    if ((string.IsNullOrEmpty(GetFAndC_FolderEntryUUIDText.Text) && string.IsNullOrEmpty(GetFAndC_FolderUniqueIDText.Text))
                     || (!string.IsNullOrEmpty(GetFAndC_FolderEntryUUIDText.Text) && (!string.IsNullOrEmpty(GetFAndC_FolderUniqueIDText.Text))))
                        throw new Exception("Must have one and only one Value");

                    if (!string.IsNullOrEmpty(GetFAndC_FolderEntryUUIDText.Text))
                        q.FolderEntryUUID.Add(GetFAndC_FolderEntryUUIDText.Text);

                    if (!string.IsNullOrEmpty(GetFAndC_FolderUniqueIDText.Text))
                        q.FolderUniqueId.Add(GetFAndC_FolderUniqueIDText.Text);
                    break;

                case QueryType.GetFoldersForDocument: //DocumentEntryEntryUUID or DocumentEntryUniqueId MUST be provide for GetFoldersForDocument
                    if ((string.IsNullOrEmpty(GetFFDoc_DocumentEntryEntryUUIDText.Text) && string.IsNullOrEmpty(GetFFDoc_DocumentEntryUniqueIdText.Text))
                    || (!string.IsNullOrEmpty(GetFFDoc_DocumentEntryEntryUUIDText.Text) && (!string.IsNullOrEmpty(GetFFDoc_DocumentEntryUniqueIdText.Text))))
                        throw new Exception("Must have one and only one Value");

                    if (!string.IsNullOrEmpty(GetFFDoc_DocumentEntryEntryUUIDText.Text))
                        q.DocumentEntryEntryUUID.Add(new XdsObjectRef(GetFFDoc_DocumentEntryEntryUUIDText.Text));

                    if (!string.IsNullOrEmpty(GetFFDoc_DocumentEntryUniqueIdText.Text))
                        q.DocumentEntryUniqueId.Add(GetFFDoc_DocumentEntryUniqueIdText.Text);
                    break;

                case QueryType.GetRelatedDocuments: // DocumentEntryEntryUUID or DocumentEntryUniqueId MUST be provided for GetRelatedDocuments
                    if ((string.IsNullOrEmpty(GetRDoc_DocumentEntryEntryUUIDText.Text) && string.IsNullOrEmpty(GetRDoc_DocumentEntryUniqueIdText.Text))
                   || (!string.IsNullOrEmpty(GetRDoc_DocumentEntryEntryUUIDText.Text) && (!string.IsNullOrEmpty(GetRDoc_DocumentEntryUniqueIdText.Text))))
                        throw new Exception("Must have one and only one Value");

                    if (!string.IsNullOrEmpty(GetRDoc_DocumentEntryEntryUUIDText.Text))
                        q.DocumentEntryEntryUUID.Add(GetRDoc_DocumentEntryEntryUUIDText.Text);

                    if (!string.IsNullOrEmpty(GetRDoc_DocumentEntryUniqueIdText.Text))
                        q.DocumentEntryUniqueId.Add(GetRDoc_DocumentEntryUniqueIdText.Text);

                    q.AssociationTypes.Add(assocType);
                    break;

                case QueryType.FindFolders:
                    //  q.FolderLastUpdateTimeFrom = new DateTime(2013, 01, 01);//optional
                    // q.FolderLastUpdateTimeTo   = new DateTime(2013, 05, 08);//optional
                    // q.PatientId = "e6b7e7213b14472^^^&1.3.6.1.4.1.21367.2005.3.7&ISO";
                    //q.FolderStatus = AvailabilityStatusCode.Approved;
                    break;
            }

            XdsQueryResponse resp;

            if (XCACheck.Checked) // If we are doing Cross Community Access Queries
            {
                // Setup the Initiating Gateway
                // Consumer talks to Initiating Gateway which talks to Responding Gateway (in a different domain) 
                // which talks to the Registry it's linked with
                xds.RegistryEndpoint = new WebServiceEndpoint(XCAInitiatingGatewayForQuery);
            }
            else // As far as the Consumer is concerned, there is NO difference between a local registry and an XCA Initiating Gateway
            {
                xds.RegistryEndpoint = new WebServiceEndpoint(RegistryAddressText.Text);
            }

            // Do the Query
            resp = xds.RegistryStoredQuery(q);

            // Display the query response
            DisplayQueryResponse(resp, q.QueryType, PIDText.Text);
        }

        /// <summary>
        /// Retrive the Document from NIST Public Repository and Display it
        /// </summary>        
        private void Retrieve(XdsDocument doc)
        {
            // Set up the retrieve request from the XdsDocument
            XdsRetrieveRequest rq = new XdsRetrieveRequest();
            rq.Add(doc);

            // If we are doing XCA Retrieve
            if (XCACheck.Checked)
                xds.RetrieveGateway = new WebServiceEndpoint(XCAInitiatingGatewayForRetrieve);

            // Retrive the document
            XdsRetrieveResponse resp = xds.RetrieveDocumentSet(rq, doc.PatientInfo);
            
            // If it's DICOM, then you may want to reference DicomObjects
            if (resp.Documents[0].MimeType.ToUpper() == "APPLICATION/DICOM")
                MessageBox.Show("If the Document is DICOM, then you need a copy of DicomObjects to handle it.");
            else // Display the downloaded Document into our Document Viewer
                xdsDocumentViewer1.Display(resp.Documents[0]);
        }

        #endregion
    }
}
