﻿using System;
using System.Collections.Generic;
using System.Windows.Forms;
using DicomObjects;
using System.Web.Http.SelfHost;
using DicomObjects.DicomWeb;
using System.Linq;
using DicomObjects.Enums;
using System.IO;
using DicomObjects.DicomWeb.WebClient;

namespace DicomWebDemo
{
    /// <summary>
    /// Use our public WADO URI https://www.dicomserver.co.uk/wado to get correct Patient ID and Study/Series/Intance UIDs for test
    /// </summary>

    public partial class Form1 : Form
    {
        internal static string studyUID = "1.3.51.0.7.1973434755.55968.57929.33465.8874.51341.44390";
        internal static string seriesUID = "1.3.51.0.7.1844502450.58935.41028.43464.58645.1281.833";
        internal static string instanceUID = "1.3.51.0.7.11390228615.60283.28999.32866.22064.32617.33089";

        readonly DicomServer InternalServer;
        public static string BaseUri = "http://localhost:9977";

        public Form1()
        {
            InitializeComponent();
            mainform = this;
            DicomGlobal.IndentJson = true;
            InternalServer = new DicomServer();
            InternalServer.Listen(888);
            InternalServer.DefaultStatus = 0;
            InternalServer.InstanceReceived += (x, y) => dicomViewer1.Images.Add(new DicomImage(y.Instance) { CineMode = CineMode.Repeat });
        }

        static Form1 mainform;
        HttpSelfHostServer listen;
        readonly DicomWebServer server = new DicomWebServer();

        private void Form1_Load(object sender, EventArgs e)
        {
            HttpSelfHostConfiguration config = new HttpSelfHostConfiguration(BaseUri);

            server.Configure(config, BaseUri, "stow", "qido", "wado", "Worklist","bulk");

            server.StowReceived += Server_StowReceived;
            server.QidoReceived += Server_QidoReceived;
            server.WadoReceived += Server_WadoReceived;

            listen = new HttpSelfHostServer(config);
            listen.OpenAsync().Wait();
        }

        private void Server_WadoReceived(WadoEventArgs args)
        {
            bool useFrameLevel = true; // this depends on the server

            DicomDataSetCollection results;
            DicomQuery q = args.MakeQuery(useFrameLevel);

            q.Node = Properties.Settings.Default.DICOMServerIPAddress;
            q.Port = Properties.Settings.Default.DICOMServerPortNumber;
            q.CalledAE = Properties.Settings.Default.DICOMServerAET;
            q.CallingAE = "WADO_Client";

            results = q.Get();
            args.ReturnData(results, useFrameLevel);
        }

        private void Server_QidoReceived(QidoEventArgs args)
        {
            DicomQuery q = new DicomQuery
            {
                Node = Properties.Settings.Default.DICOMServerIPAddress,
                Port = Properties.Settings.Default.DICOMServerPortNumber,
                CallingAE = "QIDO_Client",
                CalledAE = Properties.Settings.Default.DICOMServerAET,
                Root = QueryRoot.Study,
                Level = args.Level
            };

            // do DICOM C-FIND
            args.Results = q.Find(args.QueryDataSet());

            if (args.FuzzyMatching)
                args.Warnings.Add("The fuzzymatching parameter is not supported. Only literal matching has been performed.");
        }

        private void Server_StowReceived(StowEventArgs args)
        {
            // this is just a sample - sending on elsewhere
            var results = args.DataSets;

            foreach (DicomDataSet d in results)
            {
                try
                {
                    int status;
                    using (DicomAssociation cn = new DicomAssociation())
                    {
                        cn.RequestedContexts.Add(d.SOPClass);

                        cn.Open("localhost", 888, "STOW_Client", "me");

                        cn.SendInstances(d);
                        status = cn.LastStatus;
                        
                        cn.Close();
                    }

                    if (status == 0)
                        args.SetSuccess(d);
                    else // treat non-zero status as failure
                        args.SetFailed(d, status);
                }
                catch (Exception)
                {
                    args.SetFailed(d, 0xA900);
                }
            }
        }      

        // STOW
        private void StowBTN_Click(object sender, EventArgs e)
        {
            var dss = new DicomDataSetCollection();
            dss.Read(Properties.Resources.im1);
            var dwc = new StowWebClient($"{BaseUri}/stow/");
            dwc.Store(dss, DicomDataEncoding.Json, TransferSyntax.Text);
        }

        private void QidoBTN_Click(object sender, EventArgs e)
        {           
            var dwc = new QidoWebClient($"{BaseUri}/qido");
            var q = new DicomQuery
            {
                Level = QueryLevel.STUDY,
                Root = QueryRoot.Study
            };

            DicomDataSet ds = q.QueryDataSet();
            ds.Add(Keyword.PatientID, "MC_EXAMPLE_941");

            var result = dwc.Query(ds);
            MessageBox.Show($"Hits = {result.Count}");
        }

        // WADO - Native DICOM
        private void WadoNativeBTN_Click(object sender, EventArgs e)
        {            
            var dwc = new WadoWebClient($"{BaseUri}/wado");
            var result = dwc.RetrieveNative(studyUID);
            AddImagesWithCine(result);
        }

        // WADO - metadata only (with auto fetching of missing bulk items)
        private void WadoMedataBTN_Click(object sender, EventArgs e)
        {
            var dwc = new WadoWebClient($"{BaseUri}/wado")
            {
                TransferSyntaxes = new List<string> { "1.2.840.10008.1.2.1" }
            };
            var result = dwc.RetrieveMetaData(studyUID,
                seriesUID,
                instanceUID);
            AddImagesWithCine(result);
        }

        // Wado - get data first then merge (as one go - more efficient that separate steps)
        private void WadoMetaBulkBTN_Click(object sender, EventArgs e)
        {
            var dwc = new WadoWebClient($"{BaseUri}/wado");
            dwc.TransferSyntaxes[0] = "1.2.840.10008.1.2.1";
            var result = dwc.RetrieveMetaData(studyUID,
                seriesUID,
                instanceUID);

            dwc.AppendBulkData(result);
            AddImagesWithCine(result);
        }

        // WADO - rendered
        private void WadoRenderBTN_Click(object sender, EventArgs e)
        {
            var dwc = new WadoWebClient($"{BaseUri}/wado");            
            var result = dwc.RetrieveRendered(studyUID,
                seriesUID,
                instanceUID);
            
            // this is single only for now#
            var doc = result.First();

            string filename = "";
            var z = doc.Key.ContentType.MediaType;

            if (z.StartsWith("image"))
                filename = z.Substring(6);

            else if (z.StartsWith("application"))
                filename = z.Substring(12);

            else if (z.StartsWith("video"))
                filename = z.Substring(6);

            filename = "c:\\t1." + filename;
            File.WriteAllBytes(filename, doc.Value);

            webBrowser1.Navigate(filename);
        }
        public static void AddImagesWithCine(IEnumerable<DicomDataSet> results)
        {
            foreach (DicomDataSet d in results)
            {
                d.GetAllBulkData();
                var img = new DicomImage(d);
                if (img.FrameCount > 1)
                    img.CineMode = CineMode.Repeat;                
                mainform.dicomViewer1.Images.Add(img);
            }
        }

        private void Clear_Click(object sender, EventArgs e)
        {
            dicomViewer1.Images.Clear();
        }

    }
}

