using System;
using System.Drawing;
using System.Windows.Forms;
using DicomObjects;
using DicomObjects.EventArguments;
using DicomObjects.Enums;
using System.Collections.Generic;
using DicomObjects.DicomUIDs;

namespace PrintSCP
{
	/// <summary>
	/// Summary description for Form1.
	/// </summary>
	public class Form1 : Form
	{
		private TextBox LogText;
		private Label label1;
		/// <summary>
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;
		
		private DicomServer Server;
		public static DicomDataSet printerObject;
		private myPrintDoc myPrinterDoc;
		public class PrinterAssociation : DicomAssociation
		{
		    public PrinterAssociation()
			{
				DataSets = new DicomDataSetCollection();
				DataSets.Add(printerObject); // enables N-GET of printer object
			}

		    public DicomDataSetCollection DataSets { get; set; }
		}

		public class SessionDataSet : DicomDataSet
		{
		    public SessionDataSet()
			{
				FilmBoxes = new DicomDataSetCollection();
			}

		    public DicomDataSetCollection FilmBoxes { get; set; }
		}

		public class myPrintDoc : System.Drawing.Printing.PrintDocument
		{
		    public myPrintDoc(DicomDataSetCollection datasets)
			{
				PrintDataSets = datasets;
				UID = "";
				Status = 0xC000; // default value could be something more meaningful
			}                                                               

		    public DicomDataSetCollection PrintDataSets { get; set; }

		    public string UID { get; set; }

		    public int Status { get; set; }

		    public int CurrentPageNumber { get; set; }

		    public int TotalPageNumber { get; set; }
        }

		public Form1()
		{
			//
			// Required for Windows Form Designer support
			//
			InitializeComponent();
		}

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if (components != null) 
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#region Windows Form Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
            this.LogText = new TextBox();
            this.label1 = new Label();
            this.SuspendLayout();
            // 
            // LogText
            // 
            this.LogText.BorderStyle = BorderStyle.FixedSingle;
            this.LogText.Location = new System.Drawing.Point(8, 32);
            this.LogText.Multiline = true;
            this.LogText.Name = "LogText";
            this.LogText.ScrollBars = ScrollBars.Vertical;
            this.LogText.Size = new System.Drawing.Size(560, 360);
            this.LogText.TabIndex = 0;
            // 
            // label1
            // 
            this.label1.AutoSize = true;
            this.label1.Location = new System.Drawing.Point(8, 8);
            this.label1.Name = "label1";
            this.label1.Size = new System.Drawing.Size(229, 13);
            this.label1.TabIndex = 1;
            this.label1.Text = "By default, DicomServer is listening on port 104";
            // 
            // Form1
            // 
            this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
            this.ClientSize = new System.Drawing.Size(576, 398);
            this.Controls.Add(this.label1);
            this.Controls.Add(this.LogText);
            this.Name = "Form1";
            this.Text = "Print SCP C# 2003 Sample Program";
            this.FormClosing += new FormClosingEventHandler(this.Form1_FormClosing);
            this.Load += new System.EventHandler(this.Form1_Load);
            this.ResumeLayout(false);
            this.PerformLayout();

		}
		#endregion

		/// <summary>
		/// The main entry point for the application.
		/// </summary>
		[STAThread]
		static void Main() 
		{
			Application.Run(new Form1());
		}
		
		private void log(string s)
		{
			AppendText(LogText, s);
		}

		delegate void SetTextCallback(TextBox box, string text);

		private void AppendText(TextBox box, string text)
		{
			// InvokeRequired required compares the thread ID of the
			// calling thread to the thread ID of the creating thread.
			// If these threads are different, it returns true.
			if (box.InvokeRequired)
			{
				SetTextCallback d = AppendText;
				Invoke(d, new object[] { box, text });
			}
			else
			{
				box.Text = box.Text + text + "\r\n";
			}
		}

		private void DecodeFormat(string format, out int x, out int y)
		{
			int p, p1;
			p = format.IndexOf(@"\");
			p1 = format.IndexOf(",", p + 1);
			x = Convert.ToInt32(format.Substring(p + 1, p1 - p - 1));
			y = Convert.ToInt32(format.Substring(p1 + 1));
		}

		private void MakePrinterDataset()
		{
                //  Dummy values as placeholder    //
			DicomDataSet p = new DicomDataSet();
			p.InstanceUID = DicomObjects.DicomUIDs.WellKnownInstances.Printer;
			p.Add(8, 0x16, DicomObjects.DicomUIDs.SOPClasses.Printer);
			p.Add(8, 0x70, "Medical Connections");
			p.Add(8, 0x1090, "Demo Printer SCP");
			p.Add(0x2110, 0x10, "NORMAL");
			p.Add(0x2110, 0x20, "NORMAL");
			p.Add(0x18, 0x1000, "serial no 1234");
			p.Add(0x18, 0x1020, DicomGlobal.AssemblyVersion);
            p.Add(0x2110, 0x0030, "PrinterName");                               // Printer Name
            p.Add(0x0018, 0x1200, DateTime.Now.Date.ToString("yyyyMMdd"));      // Date of Last Calibration
            p.Add(0x0018, 0x1201, DateTime.Now.TimeOfDay.ToString("hhmmss"));   // Time of Last Calibration
			printerObject = p;
		}


		private void Form1_Load(object sender, EventArgs e)
		{
			int port = 104;
			Server = new DicomServer();
			try
			{
				DicomGlobal.LogToFile("c:\\dicom log files\\", 0x3F);
				Server.Listen(port);
				Server.NewAssociationType = typeof(PrinterAssociation);
				log("Server Started, Listen Port " + port.ToString());
				Server.AssociationRequest +=Server_AssociationRequest;                    
				Server.AssociationClosed +=Server_AssociationClosed;                    
				Server.NormalisedReceived +=Server_NormalisedReceived;                    
			}
			catch(DicomException ex)
			{
				log("ERROR! Failed to Start Server : " + ex.Message);
			}
			// Make a Global Printer Object
			MakePrinterDataset();

            initializeSupportedDisplayFomats();
		}

        class DisplayFormats
        {
            public DisplayFormats(int id, RectangleF area, Font fontType)
            {
                ID = id;
                Area = area;
                FontType = fontType;
            }
            public int ID;
            public RectangleF Area;
            public Font FontType;
        }

        void initializeSupportedDisplayFomats()
        {
            //  Defining SCP supported Annotation boxes
            DicomDataSet annotation1 = new DicomDataSet();
            annotation1.Add(Keyword.ReferencedSOPClassUID, SOPClasses.BasicAnnotationBox);
            annotation1.Add(Keyword.ReferencedSOPInstanceUID, DicomGlobal.NewUID());
            BasicAnnotationBoxes.Add(annotation1);

            DicomDataSet annotation2 = new DicomDataSet();
            annotation2.Add(Keyword.ReferencedSOPClassUID, SOPClasses.BasicAnnotationBox);
            annotation2.Add(Keyword.ReferencedSOPInstanceUID, DicomGlobal.NewUID());
            BasicAnnotationBoxes.Add(annotation2);


            //  Defining display formats for each AnnotationID
            SupportedDisplayFormats.Add(new DisplayFormats(1, new RectangleF(10, 10, 100, 100), new Font("Times New Roman", 10)));
            SupportedDisplayFormats.Add(new DisplayFormats(2, new RectangleF(150, 50, 100, 200), new Font("Consolas", 20)));
            SupportedDisplayFormats.Add(new DisplayFormats(3, new RectangleF(80, 150, 100, 200), new Font("Arial", 8)));
        }


		private void Server_AssociationRequest(object sender, AssociationRequestArgs e)
		{
			log("Association Request from " + e.Association.CallingAET + " At " + e.Association.RemoteIP);
			
			foreach(DicomContext ct in e.Contexts)
			{
				if(ct.AbstractSyntax == "1.2.840.10008.5.1.1.9") //Basic Grayscale Print
				{
					DicomDataSetCollection nullDataSetCollection = new DicomDataSetCollection();
					myPrinterDoc = new myPrintDoc(nullDataSetCollection);
					myPrinterDoc.CurrentPageNumber = 0;                
					myPrinterDoc.TotalPageNumber = 0;
					myPrinterDoc.PrintPage +=myPrinterDoc_PrintPage;
					myPrinterDoc.EndPrint +=myPrinterDoc_EndPrint;
				}
			}
		}

		private void Server_AssociationClosed(object sender, AssociationClosedArgs e)
		{
			log("Association " + e.Association.AssociationNumber + " Closed");
		}

        DicomDataSetCollection BasicAnnotationBoxes = new DicomDataSetCollection();
        DicomLabelCollection AnnotationLabels = null;
        List<DisplayFormats> SupportedDisplayFormats = new List<DisplayFormats>();            

		private void Server_NormalisedReceived(object sender, NormalisedReceivedArgs e)
		{
			PrinterAssociation assoc = (PrinterAssociation)e.RequestAssociation;
			DicomDataSetCollection DataSets = assoc.DataSets;

			DicomDataSet ds=null;
			DicomDataSetCollection ImageBoxes;
            DicomDataSet command = e.Command;

            int operation = Convert.ToInt32(command[0, 0x100].Value);
			string requestedSOPClass = command[0, 0x3].Value + "";	// Requested SOP Class UID
			string requestedSOPInstanceUID = command[0, 0x1001].Value + "";	// Requested SOP Instance UID
			string affectedSOPClass = command[0, 0x2].Value + "";	// Affected SOP Class UID
			string affectedSOPInstanceUID = command[0, 0x1000].Value + "";	// Affected SOP Instance UID
			string sessionUID="";
            

			switch (operation)
			{
				case 0x110: // N-GET
					ds = DataSets[requestedSOPInstanceUID];
					e.ResultDataSet = ds;
					e.Status = 0;
					break;

				case 0x140: // N-CREATE
					ds = e.Request.Clone() as DicomDataSet;	// Set Default Values					
					ds.Add(8, 0x16, affectedSOPClass);

					if (affectedSOPInstanceUID == "")
						affectedSOPInstanceUID = DicomGlobal.NewUID();
					ds.Add(8, 0x18, affectedSOPInstanceUID);

					// FilmSession
					if(affectedSOPClass == DicomObjects.DicomUIDs.SOPClasses.BasicFilmSession)
					{
						SessionDataSet sessionDS = new SessionDataSet();
						foreach(DicomAttribute attr in ds)
						{
							sessionDS.Add(attr.Group, attr.Element,attr.Value);
						}
						sessionDS.ValidationTypes = DicomObjects.Enums.ValidationTypes.None;
						DataSets.Add(sessionDS);
					}	


					// Specific to FilmBox
					if (affectedSOPClass == DicomObjects.DicomUIDs.SOPClasses.BasicFilmBox)
					{
						int cols, rows;
						DecodeFormat(ds[0x2010, 0x10].Value.ToString(), out cols, out rows);			
						ImageBoxes = new DicomDataSetCollection();

                        // e.PresentationContextId is only available in DicomObjects Version 5.7 and onwards
					    bool color = e.RequestAssociation.AgreedContexts[e.PresentationContextId].AbstractSyntax == DicomObjects.DicomUIDs.MetaSOPClasses.BasicColorPrint.ToString();

					    for (int i = 1; i <= rows * cols; i++)
						{
                            DicomDataSet ds1 = new DicomDataSet
                            {
                                InstanceUID = DicomGlobal.NewUID()
                            };
                            if (!color)
							    ds1.Add(8, 0x1150, DicomObjects.DicomUIDs.SOPClasses.BasicGrayscaleImageBox); // Referenced SOP Class UID
                            else
                                ds1.Add(8, 0x1150, DicomObjects.DicomUIDs.SOPClasses.BasicColorImageBox); // Referenced SOP Class UID

							ds1.Add(8, 0x1155, ds1.InstanceUID); // Referenced SOP Instance UID
							ImageBoxes.Add(ds1);
							DataSets.Add(ds1);
						}

                        ds.Add(Keyword.ReferencedImageBoxSequence, ImageBoxes); // Referenced Image Box Sequence
                        
                        //  Check if SOP class has been proposed, only then add it.
                        foreach(var agreedContext in e.RequestAssociation.AgreedContexts)
                            if(agreedContext.AbstractSyntax == SOPClasses.BasicAnnotationBox.ToString())
                                ds.Add(Keyword.ReferencedBasicAnnotationBoxSequence, BasicAnnotationBoxes);

						// Also Add FilmBox to Session
						DicomDataSetCollection SessionSequence = (DicomDataSetCollection)ds[0x2010,0x0500].Value;
						sessionUID = SessionSequence[0][0x8,0x1155].Value.ToString();
						SessionDataSet sessionDS = DataSets[sessionUID] as SessionDataSet; // Add FilmBox to the Session
						sessionDS.FilmBoxes.Add(ds);                        
						DataSets.Add(ds);
					}								
					
					e.ResultDataSet = ds;
					e.Status = 0;
					break;

				case 0x120: // N-SET
                    foreach (DicomDataSet ba in BasicAnnotationBoxes)
                        if (ba[Keyword.ReferencedSOPInstanceUID].Value.ToString() == requestedSOPInstanceUID)
                            ds = ba;                    
                    if (ds == null)
                        ds = DataSets[requestedSOPInstanceUID];
                    else
                        generateAnnotationLabels(e.Request); 
					foreach(DicomAttribute a in e.Request)
					{
						ds.Add(a.Group, a.Element, a.Value);
					}
					e.Status = 0;
					break;

				case 0x130: // N-ACTION					
					DicomDataSet filmbox; 
					filmbox = DataSets[requestedSOPInstanceUID];

					string orientation;
					orientation = filmbox[0x2010,0x40].Value.ToString();
					if(orientation == "PORTRAIT")
					{
						myPrinterDoc.DefaultPageSettings.Landscape = false;
					}
					else
					{
						myPrinterDoc.DefaultPageSettings.Landscape = true;
					}

					if(requestedSOPClass == DicomObjects.DicomUIDs.SOPClasses.BasicFilmBox)
					{
						myPrinterDoc.UID = requestedSOPInstanceUID;
						myPrinterDoc.PrintDataSets = DataSets;
						myPrinterDoc.CurrentPageNumber += 1;
						myPrinterDoc.Print();
						e.Status = myPrinterDoc.Status;
					}
					else // Session We ASSUME
					{
						SessionDataSet sessionDS = DataSets[requestedSOPInstanceUID] as SessionDataSet;
						int status = 0xC000;
						foreach(DicomDataSet filmbox1 in sessionDS.FilmBoxes)
						{
							myPrinterDoc.UID = filmbox1.InstanceUID;
							myPrinterDoc.TotalPageNumber = sessionDS.FilmBoxes.Count;
							myPrinterDoc.CurrentPageNumber = 1;
							myPrinterDoc.PrintDataSets = DataSets;
							myPrinterDoc.Print();
							status = myPrinterDoc.Status;
						}
						e.Status = status;
					}
					break;

				case 0x150: //N-Delete
					DataSets.Remove(requestedSOPInstanceUID);
					e.Status = 0;
					break;				
			}
		}

        private void generateAnnotationLabels(DicomDataSet requestDS)
        {               
            DisplayFormats chosenDisplayFormat = null;
            foreach (var df in SupportedDisplayFormats)
            {
                if (df.ID == int.Parse(requestDS[Keyword.AnnotationPosition].Value.ToString()))
                    chosenDisplayFormat = df;

                if (chosenDisplayFormat != null)
                {
                    DicomLabel l = new DicomLabel()
                    {
                        Text = requestDS[Keyword.TextString].Value.ToString(),
                        Area = chosenDisplayFormat.Area,
                        Font = chosenDisplayFormat.FontType
                    };

                    if (AnnotationLabels == null)
                        AnnotationLabels = new DicomLabelCollection();
                    AnnotationLabels.Add(l);
                    chosenDisplayFormat = null;
                }
            }
        }        

		private void myPrinterDoc_PrintPage(object sender, System.Drawing.Printing.PrintPageEventArgs e)
		{
			log("Starting Printing");

			myPrintDoc mp = (myPrintDoc)sender;

			DicomDataSetCollection datasets = mp.PrintDataSets;
			DicomDataSet filmbox = datasets[mp.UID];
			DicomDataSetCollection dss = filmbox[0x2010,0x0510].Value as DicomDataSetCollection;
			
			DicomDataSetCollection dss1 = filmbox[0x2010,0x0500].Value as DicomDataSetCollection;
			DicomDataSet session = datasets[dss1[0][0x8,0x1155].Value.ToString()];
			
			int rows, columns;
			DecodeFormat(filmbox[0x2010,0x10].Value.ToString(), out columns, out rows);
			
			int margin = 5;
			int width, height;
			int paperWidth, paperHeight;

			if(mp.DefaultPageSettings.Landscape == false)
			{
				paperWidth = mp.PrinterSettings.PaperSizes[0].Width;
				paperHeight = mp.PrinterSettings.PaperSizes[0].Height;
			}
			else
			{
				paperWidth = mp.PrinterSettings.PaperSizes[0].Height;
				paperHeight = mp.PrinterSettings.PaperSizes[0].Width;
			}

			width = ((paperWidth - margin) / columns) - margin;
			height = ((paperHeight - margin) / rows) - margin;

			Point LeftTop = new Point();
			Font textFont = new Font("Arial", 8);
			string pageNumber = "Page " + mp.CurrentPageNumber.ToString();
			e.Graphics.DrawString(pageNumber, textFont, Brushes.Blue, 4, 4);		

			for(int row = 0; row < rows; row++)
			{
				for(int column = 0; column < columns; column++)
				{
					DicomDataSet ds2 = datasets[dss[row*columns+column][0x8,0x1155].Value.ToString()];
					DicomAttribute imageAtt;

					// if Mono Image
					if (ds2[0x2020,0x110].Exists)
					{
						imageAtt = ds2[0x2020,0x0110];
					}
						// if not mono image, then check for a color image
					else
					{
						imageAtt = ds2[0x2020, 0x0111];
					}
					
					if(imageAtt.Exists) // has real image then print
					{
						DicomDataSetCollection dss2 = (DicomDataSetCollection)imageAtt.Value;
						// Get the DicomImage to print
						DicomObjects.DicomImage dcmImage  = new DicomImage(dss2[0]);
						// Set up the Print Area						 
						int min;
						int offSetX = 0;
						int offSetY = 0;
						if (width > height)
						{
							min = height;
							offSetX = (paperWidth  - min*columns - (columns + 1)* margin) / 2;
						}
						else 
						{
							min = width;
							offSetY = (paperHeight  - min*rows - (rows + 1)* margin) / 2;
						}

						LeftTop.X = column * (min + margin) + margin + offSetX;
						LeftTop.Y = row * (min + margin) + margin + offSetY;

						Rectangle PrintArea = new Rectangle(LeftTop.X, LeftTop.Y, min,min);
                        
                        //  Add Annotation Labels if any
                        if (AnnotationLabels!=null && AnnotationLabels.Count > 0)
                        {
                            dcmImage.Labels = AnnotationLabels;
                            AnnotationLabels = null;
                        }

						// Get Bitmap Image by DicomImage.Bitmap() method
						Image printImage = dcmImage.Bitmap();

						try
						{
							// Print Image here
							e.Graphics.DrawImage(printImage, PrintArea);
							mp.Status = 0;							
						}
						catch(Exception ex)
						{
							log("Fatal Error! " + ex.Message);
							// Default Status is 0xC000
						}
					}
				}
			}
		}

		private void myPrinterDoc_EndPrint(object sender, System.Drawing.Printing.PrintEventArgs e)
		{
			log("Printing Finished");
		}

        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            Server.UnlistenAll();
        }
        
	}
}
