using System;
using System.Windows.Forms;
using DicomObjects;
using DicomObjects.DicomUIDs;
using DicomObjects.Enums;
using System.IO;

namespace KeyObjectNote
{
    public partial class Form1 : Form
    {
        private const string SelectObject_Message = "Select one or more objects first";
        private const string ValueType_Container = "CONTAINER";
        private const string ValueType_ContinuityOfContent = "SEPARATE";
        private const string ErrorMessage = "ERROR - more than one patient ID";
        private const string Path_Text = "PATH";
        private const string ClearText_Clear = "Clear";
        private const string ClearText_References = "References";
        private const string Hospital = "Hospital_";
        private const string PartOfUID = "1.2.3.4.5.6.7.8.100.";
        private const string PartOfRetrieveUrl = @"https:\\192.168.0.1\";
        private const string Modality = "KO";
        private const string CodeValue = "113030";
        private const string CodeSchemeDesignator = "DCM";
        private const string CodeMeaning = "Manifest";

        readonly DicomDataSetCollection referencedInstances = new DicomDataSetCollection();

        public Form1()
        {
            InitializeComponent();
        }

        private void Add_to_collection_Click(object sender, EventArgs e)
        {
            OpenFileDialog openFileDialog1 = new OpenFileDialog();

            var samplePath = Path.Combine(SampleExtensions.GetSampleImageFolderPath(), SampleExtensions.IMAGE_FOLDER_2D);

            if (Directory.Exists(samplePath))
                openFileDialog1.InitialDirectory = samplePath;

            openFileDialog1.Multiselect = true;
            openFileDialog1.Filter = "";

            if (openFileDialog1.ShowDialog() == DialogResult.OK)
            {
                foreach (string filename in openFileDialog1.FileNames)
                {
                    referencedInstances.Read(filename);
                }
            }
            clear.Text = $"{ClearText_Clear} {referencedInstances.Count} {ClearText_References}";
        }

        private void Generate_Key_Note_Click(object sender, EventArgs e)
        {
            SaveFileDialog dlg = new SaveFileDialog();

            if(dlg.ShowDialog()!= DialogResult.OK)
            {
                return;
            }

            DicomDataSet KON = new DicomDataSet();
            DicomDataSet FirstObject;
            if (referencedInstances.Count == 0)
            {
                MessageBox.Show(SelectObject_Message);
                return;
            }
            FirstObject = referencedInstances[0];

            // General DICOM Information
            KON.SOPClass = SOPClasses.KeyObjectSelectionDocument;
            KON.StudyUID = FirstObject.StudyUID;
            KON.SeriesUID = DicomGlobal.NewUID();
            KON.InstanceUID = DicomGlobal.NewUID();
            KON.Add(Keyword.SeriesNumber, FirstObject.Value(Keyword.SeriesNumber));

            // Module: Patient Module (M)  - all taken from FirstObject
            KON.Add(Keyword.PatientName, FirstObject.Value(Keyword.PatientName));
            KON.Add(Keyword.PatientID, FirstObject.Value(Keyword.PatientID));
            KON.Add(Keyword.PatientBirthDate, FirstObject.Value(Keyword.PatientBirthDate));
            KON.Add(Keyword.PatientSex, FirstObject.Value(Keyword.PatientSex));

            // Module: General Study Module (M) - all taken from FirstObject
            KON.Add(Keyword.StudyDate, FirstObject.Value(Keyword.StudyDate));
            KON.Add(Keyword.StudyTime, FirstObject.Value(Keyword.StudyTime));
            KON.Add(Keyword.AccessionNumber, FirstObject.Value(Keyword.AccessionNumber));
            KON.Add(Keyword.ReferringPhysicianName, FirstObject.Value(Keyword.ReferringPhysicianName));
            KON.Add(Keyword.StudyID, FirstObject.Value(Keyword.StudyID));

            // Module: General Equipment Module (M)
            KON.Add(Keyword.Manufacturer, null);
            KON.Add(Keyword.SeriesNumber, 1);

            // Module: SR Document Content Module (M)
            KON.Add(Keyword.ValueType, ValueType_Container);
            KON.Add(Keyword.ContinuityOfContent, ValueType_ContinuityOfContent);
            AddConceptNameCodeSequence(KON);
            
            // Module: SOP Common Module (M)
            KON.Add(Keyword.InstanceNumber, 123456);

            // Module: Key Object Document Module (M) 
            KON.Add(Keyword.ContentDate, DateTime.Now.Date);
            KON.Add(Keyword.ContentTime, DateTime.Now.Date);

            DicomDataSet PseudoDicomDir = new DicomDataSet();  // Reference to the objects in the KON is in a hierarchy eqivelent to a DicomDir (which DicomObjects can generat automaticaly)
            // hierarchy Study - Series - Instance
            foreach (DicomDataSet ThisObject in referencedInstances)
            {
                PseudoDicomDir.AddToDirectory(ThisObject, Path_Text, "");          // Add all the Objects to the DicomDir
            }

            if (PseudoDicomDir.Children.Count > 1)
                MessageBox.Show(ErrorMessage);        // The Key Object Note is limited to one Patient - Top level of the Dicom Dir is Patients

            DicomDataSetCollection studySequence = new DicomDataSetCollection();// create a Study Level Sequance
            foreach (DicomDataSet study in PseudoDicomDir.Children[0].Children) // foreach Study in the dicomdir (only one paient allowed)
            {
                DicomDataSet StudySequenceItem = new DicomDataSet
                {
                    StudyUID = study.StudyUID                    // Referenced Study Instance UID
                };
                DicomDataSetCollection SeriesSequence = new DicomDataSetCollection();
                foreach (DicomDataSet series in study.Children)                 // for each series in each study
                {
                    int seriesCount=1;
                    DicomDataSet SeriesSequenceItem = new DicomDataSet
                    {
                        SeriesUID = series.SeriesUID            // Referenced Series Instance UID
                    };
                    DicomDataSetCollection InstanceSequence = new DicomDataSetCollection();
                    foreach (DicomDataSet Instance in series.Children)          // for each instance in each series
                    {
                        DicomDataSet InstanceSequenceItem = new DicomDataSet
                        {
                            { Keyword.ReferencedSOPClassUID, Instance.Value(Keyword.ReferencedSOPClassUIDInFile) },
                            { Keyword.ReferencedSOPInstanceUID, Instance.Value(Keyword.ReferencedSOPInstanceUIDInFile) }
                        };
                        InstanceSequence.Add(InstanceSequenceItem);
                    }
                    SeriesSequenceItem.Add(Keyword.ReferencedSOPSequence, InstanceSequence); // add all the instance collection to the series
                    string aet = Hospital+ seriesCount.ToString();
                    string uid = PartOfUID + seriesCount.ToString();
                    SeriesSequenceItem.Add(Keyword.RetrieveAETitle, aet);
                    SeriesSequenceItem.Add(Keyword.RetrieveLocationUID, uid);
                    SeriesSequenceItem.Add(Keyword.RetrieveURL, PartOfRetrieveUrl + aet+ "\\" +series.SeriesUID);
                    SeriesSequence.Add(SeriesSequenceItem);
                }
                StudySequenceItem.Add(Keyword.ReferencedSeriesSequence, SeriesSequence);  // add the series collection to the study
                studySequence.Add(StudySequenceItem);
            }
            KON.Add(Keyword.CurrentRequestedProcedureEvidenceSequence, studySequence);  // Current Requested Procedure Evidence Sequence

            // Add Referenced Request Sequence
            DicomDataSetCollection refReqSQ = new DicomDataSetCollection();
            DicomDataSet refReqSQItem = new DicomDataSet
            {
                StudyUID = studySequence[0].StudyUID
            };
            refReqSQItem.Add(Keyword.ReferencedStudySequence, studySequence);
            refReqSQItem.Add(Keyword.AccessionNumber, FirstObject.Value(Keyword.AccessionNumber));
            refReqSQItem.Add(Keyword.PlacerOrderNumberImagingServiceRequest, "");
            refReqSQItem.Add(Keyword.FillerOrderNumberImagingServiceRequest,"");
            refReqSQItem.Add(Keyword.RequestedProcedureID, "");
            refReqSQItem.Add(Keyword.RequestedProcedureDescription, "");
            refReqSQItem.Add(Keyword.RequestedProcedureCodeSequence, new DicomDataSetCollection());
            refReqSQItem.Add(Keyword.RequestedProcedureID, "");
            refReqSQ.Add(refReqSQItem);
            KON.Add(Keyword.ReferencedRequestSequence, refReqSQ);
            // Module: Key Object Document Series Module (M)
            KON.Add(Keyword.Modality, Modality);
            KON.Add(Keyword.ReferencedPerformedProcedureStepSequence, FirstObject.Value(0x0008, 0x1111));
            KON.Add(Keyword.SeriesDate, DateTime.Now.Date);
            KON.Add(Keyword.SeriesTime, DateTime.Now.Date);

            KON.Write(dlg.FileName);
        }

        private void AddConceptNameCodeSequence(DicomDataSet kon)
        {
            
            // SEQ -(0040,A043) - Concept Name Code Sequence
            DicomDataSetCollection dataSetCollection = new DicomDataSetCollection();
            DicomDataSet dataSet = new DicomDataSet
            {
                { 0x0008, 0x0100, CodeValue },                  // Code Value
                { 0x0008, 0x0102, CodeSchemeDesignator },       // Coding Scheme Designator
                { 0x0008, 0x0104, CodeMeaning }                 // Code Meaning
            };
            dataSetCollection.Add(dataSet);
            kon.Add(Keyword.ConceptNameCodeSequence, dataSetCollection);
            
        }

        private void Clear_Click(object sender, EventArgs e)
        {
            referencedInstances.Clear();
            clear.Text = $"{ClearText_Clear} {referencedInstances.Count} {ClearText_References}";
        }
    }
}