﻿using System;
using System.Collections.Generic;
using System.Linq;
using DicomObjects;
using DicomObjects.Enums;
using System.Drawing;
using System.Windows.Forms;

namespace ECGControl
{
    public partial class MyECGControl : Control
    {
        public MyECGControl()
        {
            Background = Brushes.White;
            ThinGridPen = new Pen(Brushes.Pink, 0);
            ThickGridPen = new Pen(Brushes.PaleVioletRed, 0);
            LinePen = Pens.Blue;
            LeadFormat = LeadFormats.Format_12x1;
            Amplitude = Amplitudes.Amp_10mmPerMv;
            Speed = Speeds.Speed_25mmPerSec;
            TextFont = SystemFonts.DialogFont;
            TextBrush = Brushes.Black;
            ChannelSpacing = 15;
        }

        // public properties

        public Pen ThickGridPen { get; set; }
        public Pen ThinGridPen { get; set; }
        public Pen LinePen { get; set; }
        public LeadFormats LeadFormat { get; set; }
        public Amplitudes Amplitude { get; set; }
        public Speeds Speed { get; set; }
        public Brush Background { get; set; }
        public Font TextFont { get; set; }
        public Brush TextBrush { get; set; }
        public int ChannelSpacing { get; set; }

        Waveform waveform;

        protected override void OnPaint(PaintEventArgs e)
        {
            base.OnPaint(e);
            e.Graphics.FillRectangle(Background, ClientRectangle);
            RenderECG(e.Graphics, Size, waveform);
        }

        public void LoadECG(DicomDataSet data)
        {
            waveform = LoadWaveform(data);
        }

        static Dictionary<string, ChannelID> mapping = new Dictionary<string, ChannelID>
        {
            {"Lead I",ChannelID.I},
            {"Lead II",ChannelID.II},
            {"Lead III",ChannelID.III},
            {"Lead V1",ChannelID.V1},
            {"Lead V2",ChannelID.V2},
            {"Lead V3",ChannelID.V3},
            {"Lead V4",ChannelID.V4},
            {"Lead V5",ChannelID.V5},
            {"Lead V6",ChannelID.V6},
            {"Lead aVf",ChannelID.aVF},
            {"Lead aVl",ChannelID.aVL},
            {"Lead aVr",ChannelID.aVR},
        };

        private Waveform LoadWaveform(DicomDataSet data)
        {
            if (data[Keyword.WaveformSequence].ExistsWithValue)
            {
                DicomDataSetCollection WaveformSequence = (DicomDataSetCollection)data[Keyword.WaveformSequence].Value;
                DicomDataSet WaveformSequenceItem = WaveformSequence[0];
                Waveform waveform = new Waveform();

                if (WaveformSequenceItem[Keyword.MultiplexGroupTimeOffset].ExistsWithValue)
                    waveform.MultiplexGroupTimeOffset = Convert.ToInt32(WaveformSequenceItem[Keyword.MultiplexGroupTimeOffset].Value);

                if (WaveformSequenceItem[Keyword.TriggerTimeOffset].ExistsWithValue)
                    waveform.TriggerTimeOffset = Convert.ToInt32(WaveformSequenceItem[Keyword.TriggerTimeOffset].Value);

                if (WaveformSequenceItem[Keyword.WaveformOriginality].ExistsWithValue)
                    waveform.Originality = WaveformSequenceItem[Keyword.WaveformOriginality].Value.ToString();

                if (WaveformSequenceItem[Keyword.NumberOfWaveformChannels].ExistsWithValue)
                    waveform.NumberOfChannels = Convert.ToInt32(WaveformSequenceItem[Keyword.NumberOfWaveformChannels].Value);

                if (WaveformSequenceItem[Keyword.NumberOfWaveformSamples].ExistsWithValue)
                    waveform.NumberOfSamples = Convert.ToInt32(WaveformSequenceItem[Keyword.NumberOfWaveformSamples].Value);

                if (WaveformSequenceItem[Keyword.SamplingFrequency].ExistsWithValue)
                    waveform.SamplingFrequency = Convert.ToInt32(WaveformSequenceItem[Keyword.SamplingFrequency].Value);

                if (WaveformSequenceItem[Keyword.WaveformBitsAllocated].ExistsWithValue)
                    waveform.BitsAllocated = Convert.ToInt32(WaveformSequenceItem[Keyword.WaveformBitsAllocated].Value);

                if (WaveformSequenceItem[Keyword.WaveformSampleInterpretation].ExistsWithValue)
                    waveform.SampleInterpretation = WaveformSequenceItem[Keyword.WaveformSampleInterpretation].Value.ToString();

#warning "Padding Value?"
                //if(WaveformSequenceItem[Keyword.WaveformPaddingValue].ExistsWithValue)
                //    waveform.PaddingValue  = (long[])WaveformSequenceItem[Keyword.WaveformPaddingValue].Value;


                if (WaveformSequenceItem[Keyword.ChannelDefinitionSequence].ExistsWithValue)
                {
                    int index = 0;

                    foreach (DicomDataSet channelItem in (DicomDataSetCollection)WaveformSequenceItem[Keyword.ChannelDefinitionSequence].Value)
                    {
                        WaveformChannel channel = new WaveformChannel() { index = index++ };
                        channel.info = new ChanellInfo();

                        if (channelItem[Keyword.ChannelSourceSequence].ExistsWithValue)
                        {
                            DicomDataSetCollection sourceSequence = (DicomDataSetCollection)channelItem[Keyword.ChannelSourceSequence].Value;
                            DicomDataSet sourceSequenceItem = sourceSequence[0];
                            channel.source = new DICOMCode(sourceSequenceItem[Keyword.CodeValue].Value.ToString(),
                                                           sourceSequenceItem[Keyword.CodingSchemeDesignator].Value.ToString(),
                                                           sourceSequenceItem[Keyword.CodingSchemeVersion].Value.ToString(),
                                                           sourceSequenceItem[Keyword.CodeMeaning].Value.ToString());

                            channel.channelID = mapping[channel.source.CodeMeaning];
                        }

                        if (channelItem[Keyword.ChannelSensitivity].ExistsWithValue)
                            channel.info.Sensitivity = float.Parse(channelItem[Keyword.ChannelSensitivity].Value.ToString());

                        if (channelItem[Keyword.ChannelSensitivityUnitsSequence].ExistsWithValue)
                        {
                            DicomDataSetCollection unitsSequence = (DicomDataSetCollection)channelItem[Keyword.ChannelSensitivityUnitsSequence].Value;
                            DicomDataSet unitsSequenceItem = unitsSequence[0];
                            channel.info.SensitivityUnit = new DICOMCode(unitsSequenceItem[Keyword.CodeValue].Value.ToString(),
                                                           unitsSequenceItem[Keyword.CodingSchemeDesignator].Value.ToString(),
                                                           unitsSequenceItem[Keyword.CodingSchemeVersion].Value.ToString(),
                                                           unitsSequenceItem[Keyword.CodeMeaning].Value.ToString());
                        }

                        if (channelItem[Keyword.ChannelSensitivityCorrectionFactor].ExistsWithValue)
                            channel.info.SensitivityCorrectionFactor = Convert.ToInt32(channelItem[Keyword.ChannelSensitivityCorrectionFactor].Value);

                        if (channelItem[Keyword.ChannelBaseline].ExistsWithValue)
                            channel.info.Baseline = Convert.ToInt32(channelItem[Keyword.ChannelBaseline].Value);

                        if (channelItem[Keyword.ChannelSampleSkew].ExistsWithValue)
                            channel.info.SampleSkew = Convert.ToInt32(channelItem[Keyword.ChannelSampleSkew].Value);

                        if (channelItem[Keyword.WaveformBitsStored].ExistsWithValue)
                            channel.info.BitsStored = Convert.ToUInt16(channelItem[Keyword.WaveformBitsStored].Value);

                        if (!waveform.Channels.ContainsKey(channel.channelID))
                            waveform.Channels.Add(channel.channelID, channel);
                    }

                    switch (waveform.BitsAllocated)
                    {
                        case 8:
                            byte[] rawByteData = (byte[])WaveformSequenceItem[Keyword.WaveformData].Value;
                            foreach (KeyValuePair<ChannelID, WaveformChannel> channel in waveform.Channels)
                            {
                                channel.Value.channelData = new float[waveform.NumberOfSamples];
                                for (int sampleIndex = 0; sampleIndex < waveform.NumberOfSamples; sampleIndex++)
                                {
                                    channel.Value.channelData[sampleIndex] = rawByteData[channel.Value.index * waveform.NumberOfChannels + sampleIndex];
                                }
                            }
                            break;
                        case 16:
                            if (waveform.SampleInterpretation.Equals("SS"))
                            {
                                short[] rawSSData = (short[])WaveformSequenceItem[Keyword.WaveformData].Value;
                                foreach (KeyValuePair<ChannelID, WaveformChannel> channel in waveform.Channels)
                                {
                                    channel.Value.channelData = new float[waveform.NumberOfSamples];
                                    for (int sampleIndex = 0; sampleIndex < waveform.NumberOfSamples; sampleIndex++)
                                    {
                                        channel.Value.channelData[sampleIndex] = rawSSData[sampleIndex * waveform.NumberOfChannels + channel.Value.index];
                                    }
                                }
                            }

                            if (waveform.SampleInterpretation.Equals("US"))
                            {
                                ushort[] rawUSData = (ushort[])WaveformSequenceItem[Keyword.WaveformData].Value;
                                foreach (KeyValuePair<ChannelID, WaveformChannel> channel in waveform.Channels)
                                {
                                    channel.Value.channelData = new float[waveform.NumberOfSamples];
                                    for (int sampleIndex = 0; sampleIndex < waveform.NumberOfSamples; sampleIndex++)
                                    {
                                        channel.Value.channelData[sampleIndex] = rawUSData[channel.Value.index * waveform.NumberOfChannels + sampleIndex];
                                    }
                                }
                            }
                            break;
                        default:
                            throw new Exception("Bit depth not supported");
                    }

                    if (waveform.Channels[ChannelID.I] != null && waveform.Channels[ChannelID.II] != null)
                    {
                        foreach (ChannelID channel in new ChannelID[] { ChannelID.III, ChannelID.aVL, ChannelID.aVR, ChannelID.aVF })
                        {
                            WaveformChannel SynthesisedChannel;
                            if (!waveform.Channels.ContainsKey(channel))
                            {
                                SynthesisedChannel = new WaveformChannel
                                {
                                    source = new DICOMCode(waveform.Channels[ChannelID.I].source.CodeValue,
                                        waveform.Channels[ChannelID.I].source.CodingSchemeDesignator,
                                        waveform.Channels[ChannelID.I].source.CodingSchemeVersion,
                                        (from m in mapping where m.Value == channel select m.Key).First()
                                        ),
                                    channelData = new float[waveform.NumberOfSamples],
                                    info = waveform.Channels[ChannelID.I].info,
                                    channelID = channel
                                };

                                Func<float, float, float> synth = null;
                                switch (channel)
                                {
                                    case ChannelID.III: synth = (I, II) => II - I; break;
                                    case ChannelID.aVR: synth = (I, II) => (I + II) / -2; break;
                                    case ChannelID.aVL: synth = (I, II) => I - II / 2; break;
                                    case ChannelID.aVF: synth = (I, II) => II - I / 2; break;
                                }

                                for (int sampleIndex = 0; sampleIndex < waveform.NumberOfSamples; sampleIndex++)
                                {
                                    SynthesisedChannel.channelData[sampleIndex] = synth(waveform.Channels[ChannelID.I].channelData[sampleIndex], waveform.Channels[ChannelID.II].channelData[sampleIndex]);
                                }
                                waveform.Channels.Add(channel, SynthesisedChannel);
                            }
                        }
                    }
                }

                return waveform;
            }
            else
                return null;
        }

        private void DrawGrid(Graphics g, Size sz, Waveform waveform, SizeF scaling)
        {
            // do thins first so that thicks over-write them

            int thinGrid = 1; // mm
            int thickGrid = 5; // mm

            float usedWidth = sz.Width;
            float usedHeight = sz.Height;

            for (float x = 0; x < usedWidth; x += thinGrid * scaling.Width)
            {
                g.DrawLine(ThinGridPen, x, 0, x, usedHeight);
            }
            for (float y = 0; y < usedHeight; y += thinGrid * scaling.Height)
            {
                g.DrawLine(ThinGridPen, 0, y, usedWidth, y);
            }
            for (float x = 0; x < usedWidth; x += thickGrid * scaling.Width)
            {
                g.DrawLine(ThickGridPen, x, 0, x, usedHeight);
            }
            for (float y = 0; y < usedHeight; y += thickGrid * scaling.Height)
            {
                g.DrawLine(ThickGridPen, 0, y, usedWidth, y);
            }
        }

        private void RenderECG(Graphics g, Size sz, Waveform waveform)
        {
            // This might need to be more complex!
            int nWide = LeadFormat == LeadFormats.Format_12x1 ? 1 : 2;
            int nHigh = LeadFormat == LeadFormats.Format_12x1 ? 12 : 6;

            var sizingChannel = waveform.Channels.Values.First();

            float WidthInMm = waveform.NumberOfSamples * SpeedInMsPerS() / waveform.SamplingFrequency * nWide;
            float HeightInMm = (nHigh + 1) * ChannelSpacing;

            float scale = Math.Min(Size.Width / WidthInMm, Size.Height / HeightInMm);

            SizeF scaling = new SizeF(scale, scale);

            DrawGrid(g, sz, waveform, scaling);

            List<WaveformChannel> sortedList = waveform.Channels.Values.OrderBy(cl => cl.channelID).ToList();
            DrawChannels(g, waveform, Size, sortedList, scaling, nWide, nHigh);

            return;
        }

        // scaling is pixels per mm for X & Y - both related to "conventional" ECG display

        private float AmpInMsPerS()
        {
            switch (Amplitude)
            {
                case Amplitudes.Amp_10mmPerMv:
                    return 10;
                case Amplitudes.Amp_20mmPerMv:
                    return 20;
            }
            return 10; // default
        }

        private float SpeedInMsPerS()
        {
            switch (Speed)
            {
                case Speeds.Speed_25mmPerSec:
                    return 25;
                case Speeds.Speed_50mmPerSec:
                    return 50;
            }
            return 25; // default
        }

        private void DrawChannels(Graphics g, Waveform waveform, Size size, List<WaveformChannel> channels, SizeF Scaling, int nWide, int nHigh)
        {
            float amp = AmpInMsPerS();
            float speed = SpeedInMsPerS();

            for (int channelIndex = 0; channelIndex < channels.Count; channelIndex++)
            {
                float baseX = (channelIndex / nHigh) * (Size.Width / nWide);
                float baseY = ((channelIndex % nHigh) + 1) * ChannelSpacing * Scaling.Height;

                PointF[] points = new PointF[channels[channelIndex].channelData.Length];
                g.DrawString(channels[channelIndex].source.CodeMeaning.Replace("Lead ", ""), TextFont, TextBrush, new PointF(baseX + 5, baseY - Font.Size - ChannelSpacing * Scaling.Height / 3));
                for (int sampleIndex = 0; sampleIndex < channels[channelIndex].channelData.Length; sampleIndex++)
                {
                    float y = channels[channelIndex].channelData[sampleIndex]
                        * channels[channelIndex].info.Sensitivity
                        * channels[channelIndex].info.SensitivityCorrectionFactor * amp * Scaling.Height / 1000;

                    float x = (float)sampleIndex / waveform.SamplingFrequency * speed * Scaling.Width;
                    points[sampleIndex] = new PointF(baseX + x, baseY - y);
                }
                g.DrawLines(LinePen, points);
            }
            return;
        }

        internal class Waveform
        {
            public int MultiplexGroupTimeOffset { get; set; }
            public int TriggerTimeOffset { get; set; }
            public string Originality { get; set; }
            public int NumberOfChannels { get; set; }
            public int NumberOfSamples { get; set; }
            public int SamplingFrequency { get; set; }

            public Dictionary<ChannelID, WaveformChannel> Channels { get; set; }

            public int BitsAllocated { get; set; }
            public string SampleInterpretation { get; set; }
            public long[] PaddingValue { get; set; }

            public Waveform()
            {
                Channels = new Dictionary<ChannelID, WaveformChannel>();
            }
        }

        internal class ChanellInfo
        {
            public float Sensitivity { get; set; }
            public DICOMCode SensitivityUnit { get; set; }
            public int SensitivityCorrectionFactor { get; set; }
            public int Baseline { get; set; }
            public int SampleSkew { get; set; }
            public ushort BitsStored { get; set; }
        }

        internal class WaveformChannel
        {
            public DICOMCode source { get; set; }
            public ChannelID channelID { get; set; }
            public ChanellInfo info { get; set; }
            public int index { get; set; }
            public float[] channelData { get; set; }
        }

        internal enum ChannelID
        {
            I = 1,
            II = 2,
            III = 3,
            aVR = 4,
            aVL = 5,
            aVF = 6,
            V1 = 7,
            V2 = 8,
            V3 = 9,
            V4 = 10,
            V5 = 11,
            V6 = 12
        }

        internal class DICOMCode
        {
            public string CodeValue { get; set; }
            public string CodingSchemeDesignator { get; set; }
            public string CodingSchemeVersion { get; set; }
            public string CodeMeaning { get; set; }

            public DICOMCode(string value, string scheme, string version, string meaning)
            {
                CodeValue = value;
                CodingSchemeDesignator = scheme;
                CodingSchemeVersion = version;
                CodeMeaning = meaning;
            }
        }

        public enum LeadFormats
        {
            Format_12x1,
            Format_6x2
        }

        public enum Amplitudes
        {
            Amp_10mmPerMv,
            Amp_20mmPerMv
        }

        public enum Speeds
        {
            Speed_25mmPerSec,
            Speed_50mmPerSec
        }
    }
}
