﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Globalization;
using System.Linq;

namespace Simple3DViewer
{
    public class PresetCalculation
    {

        public static VolumeRenderingPreset Parse(dynamic jsonPreset)
        {
            var preset = new VolumeRenderingPreset
            {
                Name = jsonPreset.name
            };

            ParseScalarOpacity(jsonPreset.scalarOpacity.ToString(), preset);
            ParseColorTransfer(jsonPreset.colorTransfer.ToString(), preset);

            return preset;
        }

        private static void ParseScalarOpacity(string raw, VolumeRenderingPreset preset)
        {
            var tokens = raw.Split(' ', (char)StringSplitOptions.RemoveEmptyEntries);

            if (tokens.Length < 3)
                throw new FormatException("scalarOpacity must have at least one value-opacity pair.");

            for (int i = 1; i + 1 < tokens.Length; i += 2)
            {
                if (float.TryParse(tokens[i], NumberStyles.Any, CultureInfo.InvariantCulture, out float value) &&
                    float.TryParse(tokens[i + 1], NumberStyles.Any, CultureInfo.InvariantCulture, out float opacity))
                {
                    preset.OpacityPoints.Add(new ScalarOpacityPoint { Value = value, Opacity = opacity });
                }
                else
                {
                    throw new FormatException($"Invalid number at scalarOpacity[{i}] or [{i + 1}]");
                }
            }
        }


        private static void ParseColorTransfer(string raw, VolumeRenderingPreset preset)
        {
            var tokens = raw.Split(' ', (char)StringSplitOptions.RemoveEmptyEntries);

            if (tokens.Length < 4)
                throw new FormatException("colorTransfer must have at least one value-RGB triplet.");

            for (int i = 1; i + 3 < tokens.Length; i += 4)
            {
                if (float.TryParse(tokens[i], NumberStyles.Any, CultureInfo.InvariantCulture, out float value) &&
                    float.TryParse(tokens[i + 1], NumberStyles.Any, CultureInfo.InvariantCulture, out float r) &&
                    float.TryParse(tokens[i + 2], NumberStyles.Any, CultureInfo.InvariantCulture, out float g) &&
                    float.TryParse(tokens[i + 3], NumberStyles.Any, CultureInfo.InvariantCulture, out float b))
                {
                    preset.ColorPoints.Add(new ColorTransferPoint { Value = value, R = r, G = g, B = b });
                }
                else
                {
                    throw new FormatException($"Invalid colorTransfer value at [{i}–{i + 3}]");
                }
            }
        }



        public static Color[] GenerateLUT(VolumeRenderingPreset preset, int size = 4096)
        {
            Color[] lut = new Color[size];
            float min = preset.MinValue;
            float max = preset.MaxValue;
            float range = max - min;

            for (int i = 0; i < size; i++)
            {
                float value = min + (i / (float)(size - 1)) * range;

                var color = InterpolateColor(preset.ColorPoints, value);
                var opacity = InterpolateOpacity(preset.OpacityPoints, value);

                lut[i] = Color.FromArgb((int)(opacity * 255), color.R, color.G, color.B);
            }

            return lut;
        }

        private static Color InterpolateColor(List<ColorTransferPoint> points, float value)
        {
            var before = points.LastOrDefault(p => p.Value <= value);
            var after = points.FirstOrDefault(p => p.Value >= value);

            if (before == null) return after?.ToColor() ?? Color.Black;
            if (after == null) return before.ToColor();

            if (before.Value == after.Value) return before.ToColor();

            float t = (value - before.Value) / (after.Value - before.Value);
            float r = before.R + t * (after.R - before.R);
            float g = before.G + t * (after.G - before.G);
            float b = before.B + t * (after.B - before.B);

            return Color.FromArgb((int)(r * 255), (int)(g * 255), (int)(b * 255));
        }

        private static float InterpolateOpacity(List<ScalarOpacityPoint> points, float value)
        {
            var before = points.LastOrDefault(p => p.Value <= value);
            var after = points.FirstOrDefault(p => p.Value >= value);

            if (before == null) return after?.Opacity ?? 0;
            if (after == null) return before.Opacity;

            if (before.Value == after.Value) return before.Opacity;

            float t = (value - before.Value) / (after.Value - before.Value);
            return before.Opacity + t * (after.Opacity - before.Opacity);
        }

    }
}
