﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Windows.Forms;
using DicomObjects;
using DicomObjects.Enums;

namespace Simple3DViewer
{
    public partial class Dicom3DViewer : Form
    {
        private DicomViewer viewer;
        private DicomImage3D mprAxial, mprCoronal;
        private bool draggingHandle = false;
        private float startThickness;
        private int dragDirection = 0;
        private DicomLabel slabUpperLine, slabLowerLine, slabUpperHandle, slabLowerHandle;
        private List<VolumeRenderingPreset> vtkPresets = new();
        private ComboBox presetSelector;

        const float HANDLE_SIZE = 0.02f;

        public Dicom3DViewer()
        {
            InitializeComponent();
            InitializeViewer();
        }

        private void InitializeViewer()
        {
            viewer = new DicomViewer
            {
                Dock = DockStyle.None,
                AutoDisplay = true,
                Top = 250,
                Left = 0,
                Width = this.Width,
                Height= this.Height - 300,
            };

            viewer.MouseDown += Viewer_MouseDown;
            viewer.MouseMove += Viewer_MouseMove;
            viewer.MouseUp += Viewer_MouseUp;

            Controls.Add(viewer);

            Button loadButton = new Button() { Text = "Load 3D Images", Dock = DockStyle.None, Left = 300, Width = 500, Height = 50, Top = 10 };
            loadButton.Click += LoadImages;

            Button modeButton = new Button() { Text = "Toggle MIP/Average", Dock = DockStyle.None, Left = 300, Width = 500, Height = 50, Top = 70 };
            modeButton.Click += ToggleRenderMode;

            Button vrButton = new Button() { Text = "Volume Rendering", Dock = DockStyle.None, Left = 300, Width = 500, Height = 50, Top = 140 };
            vrButton.Click += VRBTN_Click;

            Controls.Add(loadButton);
            Controls.Add(modeButton);
            Controls.Add(vrButton);
            Controls.Add(new Label { Text = "Presets: ", Dock = DockStyle.None, Left = 300, Width = 500, Height = 50, Top = 200 });

            presetSelector = new ComboBox
            {
                Dock = DockStyle.None,
                DropDownStyle = ComboBoxStyle.DropDownList,
                Left = 800,
                Top = 200,
                Width = 150,
            };
            presetSelector.SelectedIndexChanged += ApplySelectedPreset;
            Controls.Add(presetSelector);

            // Now it's safe to load and populate
            string jsonText = File.ReadAllText(@"..\..\presets.json");
            var array = Newtonsoft.Json.JsonConvert.DeserializeObject<List<dynamic>>(jsonText);
            vtkPresets = array.Select(PresetCalculation.Parse).ToList();

            presetSelector.Items.Clear();
            presetSelector.Items.AddRange(vtkPresets.Select(p => p.Name).ToArray<object>());
        }

        private void ApplySelectedPreset(object sender, EventArgs e)
        {
            if (presetSelector.SelectedIndex < 0 || mprAxial == null)
                return;

            var selectedPreset = vtkPresets[presetSelector.SelectedIndex];

            DicomImage3D vr = new DicomImage3D(mprAxial.Volume, RenderMode3D.VR);
            var lut = PresetCalculation.GenerateLUT(selectedPreset);

            vr.SetTransferFunction(selectedPreset.MinValue, selectedPreset.MaxValue, lut);
            viewer.Images.Clear();
            viewer.Images.Add(vr);
            viewer.SafeUpdate();
        }


        private void LoadImages(object sender, EventArgs e)
        {
            OpenFileDialog ofd = new OpenFileDialog() { Multiselect = true };

            if (ofd.ShowDialog() == DialogResult.OK)
            {
                var collection = new DicomImageCollection();
                collection.Read(ofd.FileNames);

                var volume = new DicomVolume(collection);

                mprCoronal = new DicomImage3D(volume, RenderMode3D.MPR);
                mprCoronal.SetViewPlane(Plane.Coronal, true, 1);

                mprAxial = new DicomImage3D(volume, RenderMode3D.MPR)
                {
                    RenderMode = RenderMode3D.Average,
                    SlabThickness = 5
                };
                mprAxial.SetViewPlane(Plane.Axial, true, 1);

                viewer.Images.Clear();
                viewer.MultiColumns = 2;
                viewer.Images.Add(mprCoronal);
                viewer.Images.Add(mprAxial);

                mprCoronal.Labels.Add(new DicomLabel()
                {
                    LabelType = LabelType.ReferenceLine,
                    RefImage = mprAxial,
                    Pen = Pens.Red
                });

                mprAxial.Labels.Add(new DicomLabel()
                {
                    LabelType = LabelType.ReferenceLine,
                    RefImage = mprCoronal,
                    Pen = Pens.Yellow
                });

                slabUpperLine = new DicomLabel()
                    { LabelType = LabelType.PolyLine, ScaleMode = ScaleMode.Cell, Pen = Pens.Cyan };
                slabLowerLine = new DicomLabel()
                    { LabelType = LabelType.PolyLine, ScaleMode = ScaleMode.Cell, Pen = Pens.Cyan };
                ;
                mprCoronal.Labels.Add(slabUpperLine);
                mprCoronal.Labels.Add(slabLowerLine);

                viewer.CurrentIndex = 0;

                 AddSlabHandles();
                 SyncSlabUI();
            }
        }

        void SyncSlabUI()
        {
            if (mprAxial == null || slabUpperLine == null || slabLowerLine == null || slabUpperHandle == null || slabLowerHandle == null)
                return;

            float slabThicknessMM = mprAxial.SlabThickness;
            float cellHeightMM = mprAxial.CellSize(viewer).Height;

            // Convert slab thickness from mm to normalized cell-space height
            float slabThicknessCell = slabThicknessMM / cellHeightMM;
            float halfSlabCell = slabThicknessCell / 2f;

            float centerY = 0.5f; // midpoint of cell vertically

            float upperY = centerY - halfSlabCell;
            float lowerY = centerY + halfSlabCell;

            // Clamp to keep handles visible in cell space (0.02 to 0.98)
            upperY = Math.Max(0.02f, upperY);
            lowerY = Math.Min(0.98f, lowerY);

            slabUpperLine.Points = new[] { new PointF(0, upperY), new PointF(1, upperY) };
            slabLowerLine.Points = new[] { new PointF(0, lowerY), new PointF(1, lowerY) };

            MoveHandle(slabUpperHandle, upperY);
            MoveHandle(slabLowerHandle, lowerY);
        }


        private void ToggleRenderMode(object sender, EventArgs e)
        {
            mprAxial.RenderMode = mprAxial.RenderMode == RenderMode3D.Average ? RenderMode3D.Maximum : RenderMode3D.Average;
            mprAxial.SlabThickness = mprAxial.RenderMode == RenderMode3D.Average ? 5 : 0;
            viewer.SafeUpdate();
        }

        private void VRBTN_Click(object sender, EventArgs e)
        {
            if (mprAxial != null)
            {
                DicomImage3D vr = new DicomImage3D(mprAxial.Volume, RenderMode3D.VR);
                viewer.Images.Clear();
                viewer.Images.Add(vr);
                return;
                // apply color transformation

                Color[] skinTF = new Color[1000];
                for (int i = 0; i < skinTF.Length; i++)
                {
                    if (i > 500) // -150 HU
                        skinTF[i] = Color.FromArgb(255, 171, 112, 53);  // skin
                }

                Color[] boneTF = new Color[1000];
                for (int i = 0; i < boneTF.Length; i++)
                {
                    if (i > 100)
                        boneTF[i] = Color.White; // bone
                }

                var colorTable = skinTF.Concat(boneTF).ToList();

                vr.ColourTable = colorTable;
                vr.SetTransferFunction(0, 2000, colorTable.ToArray());


                viewer.AdjustMultiRowsColumns();
            }
        }

        void AddSlabHandles()
        {
            slabUpperHandle = MakeHandle();
            slabLowerHandle = MakeHandle();

            mprCoronal.Labels.Add(slabUpperHandle);
            mprCoronal.Labels.Add(slabLowerHandle);

            SyncHandles();
        }

        DicomLabel MakeHandle() =>
            new DicomLabel
            {
                LabelType = LabelType.Ellipse,
                ScaleMode = ScaleMode.Cell,
                Area = new RectangleF(0, 0, HANDLE_SIZE, HANDLE_SIZE),
                Pen = Pens.Red,
                Brush = Brushes.Red,
            };

        void SyncHandles()
        {
            float centerY = 0.5f;
            float halfSlab = (mprAxial.SlabThickness / mprAxial.CellSize(viewer).Height) / 2;

            MoveHandle(slabUpperHandle, centerY - halfSlab);
            MoveHandle(slabLowerHandle, centerY + halfSlab);
        }

        void MoveHandle(DicomLabel handle, float yPosition)
        {
            float halfSize = HANDLE_SIZE / 2;
            handle.Area = new RectangleF(0.5f - halfSize, yPosition - halfSize, HANDLE_SIZE, HANDLE_SIZE);
        }

        private void Viewer_MouseDown(object sender, MouseEventArgs e)
        {
            var hits = viewer.LabelHits(e.Location, false, true);

            if (hits.Contains(slabUpperHandle))
            {
                draggingHandle = true;
                dragDirection = -1;
                startThickness = mprAxial.SlabThickness;
            }
            else if (hits.Contains(slabLowerHandle))
            {
                draggingHandle = true;
                dragDirection = 1;
                startThickness = mprAxial.SlabThickness;
            }
        }

        private void Viewer_MouseMove(object sender, MouseEventArgs e)
        {
            if (!draggingHandle) return;

            float deltaY = (e.Y - viewer.ClientSize.Height / 2f) / viewer.ClientSize.Height;
            float changeMM = deltaY * mprAxial.CellSize(viewer).Height * 2 * dragDirection;

            mprAxial.SlabThickness = Math.Max(0.1f, startThickness + changeMM);

            SyncHandles();
            SyncSlabUI();
            viewer.SafeUpdate();
        }

        private void Viewer_MouseUp(object sender, MouseEventArgs e)
        {
            draggingHandle = false;
        }
    }
}