﻿using System;
using System.Collections.Generic;
using System.IO;
using CharLS.Native;
using DicomObjects;
using DicomObjects.DicomCodecs;
using DicomObjects.DicomUIDs;
using DicomObjects.Enums;
using DicomObjects.Internal;
using Microsoft.Toolkit.HighPerformance;

namespace CustomCodec_CharLS
{
    internal partial class Program
    {
        internal class CharLsCodecFactory : CodecFactory
        {
            public override string[] SupportedTransferSyntaxes(CodecOperation operation)
            {
                // A list of Transfer Syntaxes your own codec will support
                switch (operation)
                {
                    case CodecOperation.Decompress:
                    case CodecOperation.Compress:
                        return new string[] { TransferSyntaxes.JPEGLSLossless, TransferSyntaxes.JPEGLSLossy };
                    default:
                        return Array.Empty<string>();
                }
            }

            public override IDecompressor Decompressor(string TransferSyntax, Stream Data, DicomDataSet DataSet, int Frame)
            {
                return new CharLsCodec();
            }

            public override ICompressor Compressor(string TransferSyntax)
            {
                return new CharLsCodec();
            }

            public override IImporter Importer(string Format)
            {
                throw new NotImplementedException();
            }

            public override IExporter Exporter(string Format)
            {
                throw new NotImplementedException();
            }

            public override bool CheckStart(Stream stream, string TransferSyntax)
            {
                return true;
            }
        }

        internal class CharLsCodec : ICompressor, IDecompressor
        {
            public object LockObject()
            {
                return this;
            }

            public bool NeedsPlanarConfigTransform => false;

            #region Decompressor

            public void Decompress(DecompressionArguments args)
            {
                // Plug-in CharLS Decompression code here
                var srcBytes = new byte[args.SourceStream.Length];
                args.SourceStream.Read(srcBytes, 0, srcBytes.Length);
                using JpegLSDecoder decoder = new(srcBytes);
                var decodedBytes = decoder.Decode();
                Buffer.BlockCopy(decodedBytes, 0, args.DestinationArray, 0, decodedBytes.Length);
            }

            public string DecompressedPhotometricInterpretation(DicomDataSet parent)
            {
                return parent[Keyword.PhotometricInterpretation].Value.ToString();
            }

            public PixelRequest BestRequest(float Zoom, RequestType Sync, DicomDataSet parent, int Frame)
            {
                return null;
            }

            public DecompressStatus DecompressState(PixelRequest Request)
            {
                throw new NotImplementedException();
            }

            public ProgressiveStatus DownloadStatus(PixelRequest Request)
            {
                throw new NotImplementedException();
            }

            public int CalculatedFrameCount()
            {
                throw new NotImplementedException();
            }

            #endregion

            #region Compressor

            public int MaximumBits => 16;

            public void Compress(CompressionArguments args)
            {
                // Plug-in CharLS Compression code here
                var srcLengthInBytes = args.SourceArray switch
                {
                    byte[] => args.SourceArray.Length,
                    ushort[] => args.SourceArray.Length * 2,
                    uint[] => args.SourceArray.Length * 4,
                    _ => 0
                };

                var srcBytes = new byte[srcLengthInBytes];
                Buffer.BlockCopy(args.SourceArray, 0, srcBytes, 0, srcLengthInBytes);

                using JpegLSEncoder encoder = new(new FrameInfo(args.Size.Width, args.Size.Height, args.BitsAllocated, args.Planes));

                encoder.WriteComment($"Compressed using charls [version-number] in {DicomGlobal.FileVersion}");
                encoder.Encode(srcBytes);
                encoder.EncodedData.AsStream().CopyTo(args.DestinationStream);
            }

            public void PrepareFunction(ModificationArguments args)
            {
                // Make any changes to the DicomDataSet which are required for compatibility with the 
                // compressed data, e.g.the BitDepth / Photometric Interpretation may need altering
            }

            public object Quality(string TransferSyntax, Dictionary<string, object> qmap, DicomDataSet dataset)
            {
                return null;
            }

            #endregion
        }
    }
}