﻿using System;
using System.Collections;
using System.Threading;
using DicomObjects;
using DicomObjects.Enums;
using DicomObjects.EventArguments;
using DicomObjects.UIDs;
using Microsoft.Extensions.Logging;

namespace SCP
{
    class Program
    {
		private const string PressAnyKeyToExit = "Press any key to exit";
		private const string Node = "localhost";
		private const string CallingAE = "SC_SCU";
		private const string CalledAE = "SC_SCP";
		private const int Port = 105;
		private const string IncomingOperationReceivedMessage = "Incoming Operation Received :";
		private const string InvalidCommand = "Invalid Command -";
		private const string EventReportSentMessage = "N-EVENT-REPORT Sent";

		internal static ArrayList requests;
		internal static Timer timer;
		
		internal static bool simulateFailure;
		static void Main(string[] args)
        {
			ILoggerFactory loggerFactory = LoggerFactory.Create(config => config.AddConsole());
			ILogger logger = loggerFactory.CreateLogger<Program>();
			DicomGlobal.LogToLogger(logger, 0x3F);

			DicomServer Server = new DicomServer();
            Server.NormalisedReceived += Server_NormalisedReceived;
            Server.DefaultStatus = 0xC000;
            Server.Listen(104);

            requests = new ArrayList();

			timer = new Timer(new TimerCallback(MyTimer_Elapsed), null, int.MaxValue, 2000);

			DicomGlobal.Log(PressAnyKeyToExit);
			Console.ReadKey();            
        }

        private static void Server_NormalisedReceived(object Sender, NormalisedReceivedArgs e)
        {
			string requestedSOPInstanceUID;
			long CommandField = Convert.ToInt64(e.Command[Keyword.CommandField].Value);
			int Status = 0xC000;
			string EventTypeID;

			Log($"{IncomingOperationReceivedMessage} {e.Operation}");
			switch (CommandField)
			{
				case 0x130: // N-ACTION
					requestedSOPInstanceUID = e.Command[Keyword.RequestedSOPInstanceUID].Value.ToString();
					EventTypeID = e.Command[Keyword.ActionTypeID].Value.ToString();
					// If not Storage Commitment Push then Error out
					if (requestedSOPInstanceUID != WellKnownInstances.StorageCommitmentPushModelInstance)
					{
						Status = 0x112; // Invalid object instance
					}
					else if (EventTypeID != "1")
					{
						Status = 0x123; // Invalid Action Type: 1 For Storage Commitment Request
					}
					else
					{
						// Send N-ACTION-RESP with success status
						// use RemoteAET to store the SCU's AET, which needs to be looked up from database						
						OutstandingRequest request = new OutstandingRequest
						{
							dataset = e.Request,
							RemoteAET = e.RequestAssociation.CallingAET,
							LocalAET = e.RequestAssociation.CalledAET
						};

						requests.Add(request);
						Status = 0;
						timer.Change(4000,4000);
					}
					break;
				default:
					Log($"{InvalidCommand} {CommandField.ToString("X4")}");
					break;
			}
			e.Status = Status;
		}

		private static void Log(string text)
		{
			DicomGlobal.Log(text);
		}

		private static void MyTimer_Elapsed(object obj)
		{
			// Handle a single resonse
			// for a real application this would be triggered by confirmation that the images were "safe"
			// NOT by a timer
			if (requests.Count > 0)
			{
				DicomAssociation cn2 = new DicomAssociation();
				OutstandingRequest request = requests[requests.Count - 1] as OutstandingRequest;
				DicomDataSet StorageCommitmentResult = new DicomDataSet();
				DicomDataSetCollection OK_SOP_Sequence = new DicomDataSetCollection();
				DicomDataSetCollection Failed_SOP_Sequence = new DicomDataSetCollection();
				DicomDataSetCollection refSOPSQ = (DicomDataSetCollection)request.dataset[Keyword.ReferencedSOPSequence].Value;

				bool ImageOK = true;
				// Dummy Alternation here
				// replace the following foreach loop by Image Storage Status to indicate success or failure on each Image
				foreach (DicomDataSet ds in refSOPSQ)
				{
					if (simulateFailure)
					{
						if (ImageOK)
						{
							OK_SOP_Sequence.Add(ds);
						}
						else // Simulate Image Storage Failure
						{
							// Failure Reason
							/*
                            0110H - Processing failure - A general failure in processing the operation was encountered.
                            0112H - No such object instance One or more of the elements in the Referenced SOP Instance Sequence was not available.
                            0213H - Resource limitation The SCP does not currently have enough resources to store the requested SOP Instance(s).
                            0122H - Referenced SOP Class not supported Storage Commitment has been requested for a SOP Instance with a SOP Class that is not supported by the SCP.
                            0119H - Class / Instance conflict The SOP Class of an element in the Referenced SOP Instance Sequence did not correspond to the SOP class registered for this SOP Instance at the SCP.
                            0131H - Duplicate transaction UID The Transaction UID of the Storage Commitment Request is already in use. 
                            */
							ds.Add(Keyword.FailureReason, 0x110);
							Failed_SOP_Sequence.Add(ds);
						}
						ImageOK = !ImageOK;
					}
					else
					{
						OK_SOP_Sequence.Add(ds);
					}
				}

				int EventTypeID = 1; // 1 for success and 2 for failure, according to DICOM rules

				if (OK_SOP_Sequence.Count > 0)
				{
					EventTypeID = 1;
					StorageCommitmentResult.Add(Keyword.ReferencedSOPSequence, OK_SOP_Sequence);
				}

				if (Failed_SOP_Sequence.Count > 0)
				{
					EventTypeID = 2;
					StorageCommitmentResult.Add(Keyword.FailedSOPSequence, Failed_SOP_Sequence);
				}

				StorageCommitmentResult.Add(Keyword.TransactionUID, request.dataset[Keyword.TransactionUID].Value);
				StorageCommitmentResult.Add(Keyword.RetrieveAETitle, request.LocalAET);

				cn2.RequestedContexts.Add(SOPClasses.StorageCommitmentPush);
				cn2.RequestedContexts[1].RequestorSCURole = false;
				cn2.RequestedContexts[1].RequestorSCPRole = true;

				// Lookup RemoteAET in Database to find the associated IP address and Port number.
				// In this sample program, SCP is connecting directly to the follwing SCU:
				// == AET : SC_SCU
				// == IP  : Localhost
				// == Port: 105
				cn2.Open(Node, Port, CallingAE, CalledAE);

				// Send N-EVENT-REPORT back on new Association
				cn2.NEventReport(SOPClasses.StorageCommitmentPush,
								 WellKnownInstances.StorageCommitmentPushModelInstance,
								 EventTypeID, StorageCommitmentResult);
				cn2.Close();

				Log(EventReportSentMessage);
				requests.Remove(request);
			}

			if (requests.Count > 0) // Check to see if there is another to do
			{
				timer.Change(4000,4000);
			}
			else
			{
				timer.Change(int.MaxValue, int.MaxValue); // stop timer
			}
		}
	}
}
