﻿using System;
using System.Data.SqlClient;
using System.IO;
using System.Text.RegularExpressions;
using DicomObjects;
using DicomObjects.Enums;

namespace DicomServer
{
    internal static partial class Utils
    {
        private const string Where = " Where ";
        private const string Or = " OR ";
        private const string And = " AND ";
        private const string Exists = "EXISTS";
        private const string Like = "Like";

        private const string NameLastUC = "Name_last_UC";
        private const string NameFirstUC = "Name_first_UC";
        private const string NameMiddleUC = "Name_middle_UC";
        private const string NameDICOM = "Name_DICOM";

        private const string DateTimeFormat = "yyyy/MM/dd HH:mm:ss";
        private const string DateFormat = "yyyy/MM/dd";
        private const string TimeFormat = "HH:mm:ss";
        private const string FieldName = "fieldname";
        private const string GroupId = "groupid";
        private const string ElementId = "elementid";
        private const string IsName = "isName";
        private const string Null = "NULL";
        private const string IsDate = "IsDate";
        private const string IsTime = "IsTime";
        private const string PatientId = "PatientID";
        private const string StudyUId = "StudyUID";
        private const string SeriesUId = "SeriesUID";
        private const string InstanceUId = "InstanceUID";

        private static string SplitUID(string prefix, string UID)
        {
            if (UID.Contains("\\")) // multiple UIDs
            {
                string finalstring = "";
                foreach (string uid in UID.Split(new char[] { '\\' }))
                {
                    finalstring += $"{prefix} = '{uid}' {Or} ";
                }
                return finalstring.Substring(0, finalstring.Length - 3);
            }
            else
            {
                return prefix + " = '" + UID + "' ";
            }
        }

        private static string ArrayToBackSlash(object v)
        {
            if (v == null)
                return "";
            else if (v is Array)
            {
                Array a = v as Array;
                string temp = "";
                for (int i = a.GetLowerBound(0); i <= a.GetUpperBound(0); i++)
                {
                    if (i != a.GetLowerBound(0))
                    {
                        temp += "\\";
                    }
                    temp += a.GetValue(i).ToString();
                }
                return temp;
            }
            else return v.ToString();
        }

        public static void BackSlashToArray(ref object v)
        {
            string value = v.ToString();
            if (value.IndexOf('\\') == -1) return;
            v = value.Split(new char[] { '\\' }) as object;
        }

        private static void AddWhereOrAnd(ref string Criteria, bool UseWhere, bool UseOR)
        {
            if (Criteria == "")
            {
                if (UseWhere)
                    Criteria = Where;
            }
            else if (UseOR)
            {
                Criteria += Or;
            }
            else
            {
                Criteria += And;
            }
        }

        public static void AddMatchCriterion(ref string Criteria, string FieldMatch, object val)
        {
            if (val is Array)
            {
                Array a = val as Array;
                if (a.Length > 0)
                {
                    AddWhereOrAnd(ref Criteria, true, false);
                    Criteria += " ( ";
                    for (int i = a.GetLowerBound(0); i <= a.GetUpperBound(0); i++)
                    {
                        if (i != a.GetLowerBound(0))
                            Criteria += Or;
                        Criteria += $"{Exists} ( {FieldMatch} '{StarToPercent(a.GetValue(i).ToString())}')";
                    }
                    Criteria += " ) ";
                }
            }
            else
            {
                if (val.ToString() != "" && val != null)
                {
                    AddWhereOrAnd(ref Criteria, true, false);
                    Criteria += $"{Exists} ( {FieldMatch} '{StarToPercent(val.ToString())}')";
                }
            }
        }

        public static void AddStringCriterion(ref string Criteria, string fieldname, object val, bool UseWhere, bool UseOR)
        {
            if (val is Array)
            {
                Array a = val as Array;
                if (a.GetUpperBound(0) > 0)
                {
                    AddWhereOrAnd(ref Criteria, UseWhere, UseOR);
                    Criteria = Criteria + " ( ";
                    for (int i = a.GetLowerBound(0); i <= a.GetUpperBound(0); i++)
                    {
                        if (i != a.GetLowerBound(0))
                            Criteria += Or;
                        Criteria += $"{fieldname} {Like} '{StarToPercent(a.GetValue(i).ToString())}'";
                    }
                    Criteria = Criteria + " ) ";
                }
            }
            else
            {
                if (val.ToString() != "" && val != null)
                {
                    string v = val.ToString();
                    AddWhereOrAnd(ref Criteria, UseWhere, UseOR);
                    if (v.Contains("*"))
                        Criteria += $"{fieldname} {Like} '{StarToPercent(v)}'";
                    else
                        Criteria += fieldname + " = '" + StarToPercent(v) + "'";
                }
            }
        }

        public static void AddNameCriterion(ref string Criteria, string fieldname, DicomAttribute val)
        {
            string name = val.Value.ToString() + "";
            string outerCriterion = "False";
            string title = "";
            string first = "";
            string middle = "";
            string last = "";
            string innerCriterion = "";

            if (name != "" && name != "*")
            {
                SplitName(name, ref title, ref first, ref middle, ref last);
                innerCriterion = "";
                if (last != "") AddStringCriterion(ref innerCriterion, NameLastUC, last, true, false);
                if (first != "") AddStringCriterion(ref innerCriterion, NameFirstUC, first, true, false);
                if (middle != "") AddStringCriterion(ref innerCriterion, NameMiddleUC, middle, true, false);

                if (innerCriterion == "")
                {
                    AddStringCriterion(ref innerCriterion, NameLastUC, name.Replace("^", ""), true, true);
                    AddStringCriterion(ref innerCriterion, NameFirstUC, name.Replace("^", ""), true, true);
                    AddStringCriterion(ref innerCriterion, NameMiddleUC, name.Replace("^", ""), true, true);
                    AddStringCriterion(ref innerCriterion, NameDICOM, name.Replace("^", ""), true, true);
                }

                if (innerCriterion != "")
                    outerCriterion += $"{Or} ({innerCriterion})";
                AddWhereOrAnd(ref Criteria, true, false);
            }
            if (outerCriterion != "False")
                Criteria += " " + fieldname + " IN ( select name_dicom from UnicodeNames " + innerCriterion + " ) ";
        }

        private static bool isFirstCall = true;
        private static DateTime _start = new DateTime();
        private static DateTime _finish = new DateTime();
        private static string _fieldName = "";


        //For this function to work the first call is "isDate" and the second call is "isTime" - have a look at the calling code...
        public static void AddDateTimeCriterion(ref string Criteria, string fieldname, DicomAttribute val)
        {
            if (isFirstCall)
            {
                //Remember the date
                isFirstCall = false;
                _fieldName = fieldname;
                _start = val.DateTimeFrom;
                _finish = val.DateTimeTo;

                return;
            }

            isFirstCall = true;

            //Add the time to date
            DateTime start = val.DateTimeFrom;
            DateTime finish = val.DateTimeTo;

            _start = _start.AddHours(start.Hour);
            _start = _start.AddMinutes(start.Minute);
            _start = _start.AddSeconds(start.Second);

            _finish = _finish.AddHours(finish.Hour);
            _finish = _finish.AddMinutes(finish.Minute);
            _finish = _finish.AddSeconds(finish.Second);

            Criteria += Where + _fieldName + " + " + fieldname + " >= '" + _start.ToString(DateTimeFormat) + "'" + And;
            Criteria += _fieldName + " + " + fieldname + " <= '" + _finish.ToString(DateTimeFormat) + "'";
        }

        public static void AddDateCriterion(ref string Criteria, string fieldname, DicomAttribute val)
        {
            if (val.Value != null && val.Value.ToString() != "")
            {
                DateTime start = val.DateTimeFrom;
                DateTime finish = val.DateTimeTo;

                AddWhereOrAnd(ref Criteria, true, false);

                if (start.Year < 1900)
                    start = start.AddYears(1900 - start.Year);

                Criteria = Criteria + "CAST( " + fieldname + " AS DATE) >= '" + DateToSQL(start) + "'" + And;
                Criteria = Criteria + "CAST( " + fieldname + " AS DATE) <= '" + DateToSQL(finish) + "'";
            }
        }

        public static void AddTimeCriterion(ref string Criteria, string fieldname, DicomAttribute val)
        {
            if (val.Value != null && val.Value.ToString() != "")
            {
                DateTime start = val.DateTimeFrom;
                DateTime finish = val.DateTimeTo;

                AddWhereOrAnd(ref Criteria, true, false);

                Criteria = Criteria + "CAST( " + fieldname + " AS TIME) >= '" + TimeToSQL(start) + "'" + And;
                Criteria = Criteria + "CAST( " + fieldname + " AS TIME) <= '" + TimeToSQL(finish) + "'";
            }
        }

        public static string StarToPercent(string s)
        {
            if (s.Contains("*"))
                s = s.Replace("*", "%");
            return HandleQuotes(s);
        }

        public static void SplitName(string fullname, ref string title, ref string forename, ref string middlename, ref string surname)
        {
            string name;
            if (fullname.Contains("="))
            {
                int p = (fullname + "=").IndexOf("=");
                name = fullname.Substring(0, p - 1);
            }
            else
                name = fullname.Trim();

            surname = name;

            string[] NameComponents;
            if (name.Contains("^", StringComparison.CurrentCulture)) //proper DICOM name format surname^forename^middlename^title
            {
                char[] charSeparators = new char[] { '^' };
                NameComponents = (name + "^^^^^").Split(charSeparators);
                surname = NameComponents[0];
                forename = NameComponents[1];
                middlename = NameComponents[2];
                title = NameComponents[3];
            }
            else if (name.IndexOf(",") >= 0)
            {
                char[] charSeparators = new char[] { ',' };
                NameComponents = (name + ",,,,,").Split(charSeparators);
                surname = NameComponents[0];
                forename = NameComponents[1];
                middlename = NameComponents[2];
                title = NameComponents[3];
            }
            surname = surname.Trim();
            forename = forename.Trim();
            middlename = middlename.Trim();
            title = title.Trim();
        }

        public static void MakePath(string filename)
        {
            string folder = filename.Substring(0, filename.LastIndexOf("\\"));
            if (!Directory.Exists(folder))
                Directory.CreateDirectory(folder);
        }

        public static string HandleQuotes(string a)
        {
            return Regex.Replace(a, "'", "''");
        }

        public static string DateToSQL(DateTime d)
        {
            while (d.Month <= 0)
                d.AddMonths(1);

            while (d.Month > 12)
                d.AddMonths(-1);

            while (d.Day <= 0)
                d.AddDays(1);

            while (d.Day > 31)
                d.AddDays(-1);

            return d.ToString(DateFormat);
        }

        private static string TimeToSQL(DateTime d)
        {
            return d.ToString(TimeFormat);
        }

        public static string MakeInsertString(DicomDataSet Image, SqlDataReader r, string prefix)
        {
            string name, surname = "";
            string Result = prefix + " ";
            string title = "";
            string forename = "";
            string middlename = "";
            string NameInsertString = "";
            object Value;

            while (r.Read())
            {
                name = "@" + r[FieldName].ToString();
                Value = Image[Convert.ToInt32(r[GroupId]), Convert.ToInt32(r[ElementId])].Value;
                if (Convert.ToBoolean(r[IsName]))
                {
                    string tempvalue, charset;
                    int c;
                    charset = ArrayToBackSlash(Image[0x8, 0x5].Value);
                    tempvalue = (Value == null) ? "" : Value.ToString();

                    if (Value == null)
                        Value = "";

                    c = 0;
                    if (tempvalue != "")
                    {
                        SplitName(tempvalue, ref title, ref forename, ref middlename, ref surname);
                        c += 1;
                        NameInsertString += "Execute InsertName  @charset='" + charset + "', @Name_DICOM='" + HandleQuotes(Value.ToString()) + "', @Component = " + c + ", @Name_First_UC = N'" + HandleQuotes(forename) + "',@Name_Middle_UC= N'" + HandleQuotes(middlename) + "',@Name_Last_UC = N'" + HandleQuotes(surname) + "',@Name_Title_UC = N'" + HandleQuotes(title) + "';";
                    }
                }

                Result += name + "=";
                if (Value == null)
                    Result += Null+ ", ";
                else if (Convert.ToBoolean(r[IsDate]))
                    Result += "'" + DateToSQL(Convert.ToDateTime(Value)) + "'" + ", ";
                else if (Convert.ToBoolean(r[IsTime]))
                    Result += "'" + TimeToSQL(Convert.ToDateTime(Value)) + "'" + ", ";
                else
                    Result += "'" + HandleQuotes(ArrayToBackSlash(Value)) + "'" + ", ";
            }
            Result = Result.Substring(0, Result.Length - 2);
            if (NameInsertString != "")
            {
                Result = NameInsertString + ";" + "\r\n" + Result;
            }
            return Result;
        }

        public static string MakeLegalFileName(string x)
        {
            x = Regex.Replace(x, "'", "_");
            x = Regex.Replace(x, "/", "_");
            return x;
        }

        public static SqlDataReader DoQuery(SqlConnection conn, string sql)
        {
            SqlCommand cmd = new SqlCommand(sql, conn);
            return cmd.ExecuteReader();
        }

        public static int DoCommand(SqlConnection conn, string sql)
        {
            SqlCommand cmd = new SqlCommand(sql, conn);
            return cmd.ExecuteNonQuery();
        }

        public static string LevelBasedSelection(QueryLevel level, DicomDataSet request)
        {
            switch (level)
            {
                case QueryLevel.PATIENT:
                    return Where + PatientId + " = '" + request.PatientID + "'";

                case QueryLevel.STUDY:
                    return Where + SplitUID(StudyUId, request.StudyUID);

                case QueryLevel.SERIES:
                    return Where + SplitUID(SeriesUId, request.SeriesUID);

                case QueryLevel.IMAGE:
                case QueryLevel.FRAME:
                    return Where + SplitUID(InstanceUId, request.InstanceUID);

                default:
                    return Where + SplitUID(PatientId, request.PatientID);
            }
        }

        // adds Transfer syntaxes which are not in DicomObjects default list
        public static void SetupTransferSyntaxes()
        {
            //JPIP
            DicomGlobal.AddToAcceptedTransferSyntaxes(SOPClassGroups.Images8Bit, "1.2.840.10008.1.2.4.94");
            DicomGlobal.AddToAcceptedTransferSyntaxes(SOPClassGroups.Images16Bit, "1.2.840.10008.1.2.4.94");
            DicomGlobal.AddToAcceptedTransferSyntaxes(SOPClassGroups.Images32Bit, "1.2.840.10008.1.2.4.94");
            DicomGlobal.AddToAcceptedTransferSyntaxes(SOPClassGroups.Services, "1.2.840.10008.1.2.4.94");

            DicomGlobal.AddToAcceptedTransferSyntaxes(SOPClassGroups.Images8Bit, "1.2.840.10008.1.2.4.95");
            DicomGlobal.AddToAcceptedTransferSyntaxes(SOPClassGroups.Images16Bit, "1.2.840.10008.1.2.4.95");
            DicomGlobal.AddToAcceptedTransferSyntaxes(SOPClassGroups.Images32Bit, "1.2.840.10008.1.2.4.95");

            //JPEG-LS
            DicomGlobal.AddToAcceptedTransferSyntaxes(SOPClassGroups.Images8Bit, "1.2.840.10008.1.2.4.80");
            DicomGlobal.AddToAcceptedTransferSyntaxes(SOPClassGroups.Images16Bit, "1.2.840.10008.1.2.4.80");

            DicomGlobal.AddToAcceptedTransferSyntaxes(SOPClassGroups.Images8Bit, "1.2.840.10008.1.2.4.81");
            DicomGlobal.AddToAcceptedTransferSyntaxes(SOPClassGroups.Images16Bit, "1.2.840.10008.1.2.4.81");

            // MPEG
            DicomGlobal.AddToAcceptedTransferSyntaxes(SOPClassGroups.Images8Bit, "1.2.840.10008.1.2.4.100");
            DicomGlobal.AddToAcceptedTransferSyntaxes(SOPClassGroups.Images8Bit, "1.2.840.10008.1.2.4.101");
            DicomGlobal.AddToAcceptedTransferSyntaxes(SOPClassGroups.Images8Bit, "1.2.840.10008.1.2.4.102");
            DicomGlobal.AddToAcceptedTransferSyntaxes(SOPClassGroups.Images8Bit, "1.2.840.10008.1.2.4.103");
            DicomGlobal.AddToOfferedTransferSyntaxes(SOPClassGroups.Images8Bit, "1.2.840.10008.1.2.4.100");
            DicomGlobal.AddToOfferedTransferSyntaxes(SOPClassGroups.Images8Bit, "1.2.840.10008.1.2.4.101");
            DicomGlobal.AddToOfferedTransferSyntaxes(SOPClassGroups.Images8Bit, "1.2.840.10008.1.2.4.102");
            DicomGlobal.AddToOfferedTransferSyntaxes(SOPClassGroups.Images8Bit, "1.2.840.10008.1.2.4.103");
        }
    }
    internal static class StatusCodes
    {
        // success
        public const int Success = 0x0;

        //pending
        public const int Pending = 0xFF00;

        //warnng
        public const int DuplicateInstanceReceived = 0x0111;
        public const int MoveSubOperationCompleteWithOneORMoreFailures = 0xB000;

        //errors
        public const int RefusedOutofresources = 0xA700;
        public const int MoveDestinationUnknown = 0xA801;
        public const int GeneralError = 0xC000;
        public const int ReceivedInstanceFailedInternalValidation = 0xC001;
        public const int CFindUnableToProcess = 0xC003;
        public const int CMoveUnableToProcess = 0xC004;
        public const int CGetUnableToProcess = 0xC005;
        public const int InvalidQueryRoot = 0xC006;
        public const int InvalidQueryLevel = 0xC007;
    }
}
