﻿using DicomObjects;
using DicomObjects.Enums;
using DicomObjects.UIDs;

#if WINDOWS
using Windows.Storage.Pickers;
using WinRT.Interop;
#elif MACCATALYST
using AppKit;
using Foundation;
using UIKit;
#endif
using SkiaSharp;
using System.Diagnostics;
using PointerEventArgs = Microsoft.Maui.Controls.PointerEventArgs;
using Microsoft.UI.Xaml;
using Window = Microsoft.Maui.Controls.Window;
using Application = Microsoft.Maui.Controls.Application;
using Microsoft.UI.Xaml.Input;
using Microsoft.UI;
using Microsoft.UI.Windowing;
using Windows.Graphics;
using Colors = Microsoft.Maui.Graphics.Colors;
using Microsoft.UI.Xaml.Media;

namespace MauiSampleViewer
{
    public partial class MainPage
    {
        public static MainPage? Instance { get; private set; }


        private readonly ViewerOptions _options;
        private readonly LabelForm _labelForm;
        readonly DicomServer _server;

        public MainPage() : this(MauiSampleViewer.App.SharedOptions)
        {
        }

        public MainPage(ViewerOptions options)
        {
            InitializeComponent();
            Instance = this;
            _options = options;
            _options.Viewer = Viewer;
            _labelForm = new LabelForm(_options);
            BindingContext = _options;


            _options.OnMultiRowsChanged = (rows) => Viewer.MultiRows = (short)rows;
            _options.OnMultiColumnsChanged = (columns) => Viewer.MultiColumns = (short)columns;
            Viewer.DataChanged += Viewer_DataChanged;
            _server = new DicomServer();
            _server.Listen(_options.GeneralListenPort);
            _server.InstanceReceived += (o, ee) =>
            {
                Debug.WriteLine($"Instance Received {ee.Instance.InstanceUID}");
                ee.Status = 0;
            };
            _server.VerifyReceived += (o, ee) => ee.Status = 0;
        }

        private PointF? LastPosition;
        private SKPoint MouseMoveDownPosition;

        const byte BitFlag2D = 1; // 00000001
        const byte BitFlag3D = 2; // 00000010
        const byte BitFlagBoth = 3; // 00000011

        bool LabelDrawing;
        DicomLabel? CurrentLabel, FirstLineLabel, ValueLabel, PixelValueLabel;
        ContentAlignment LabelEdge;

        private PointF? _labelStartPoint = null; // Fixed starting point of label when user begins drawing
        private const string LabelTag_Info = "INFO";
        private const string LabelTag_Ruler = "RULER";
        private const string LabelTag_Marker = "MARKER";
        private const string LabelTag_Value = "VALUE";
        private const string BottomRightLabelTagInfo = "bottomRightLabel";
        private const string LeftText = "LEFT";
        private const string TopText = "TOP";
        private const string BottomText = "BOTTOM";
        private const string RightText = "RIGHT";
        private const string frameAndZoomLabelText = $"Frame: [Frame]/[FrameCount] \r\nZoom: [$MAG]";
        private DateTime _lastInvalidate = DateTime.MinValue;
        private readonly TimeSpan _debounceInterval = TimeSpan.FromMilliseconds(100);
        private const string AnonymiserMessageBoxText = "This is for demo purpose only to show how DicomObjects can be used to anonymise DICOM instances, and if required block off any burned-in patient demographics";
        private const string AnonymiserMessageBoxTitle = "Sample Image Anonymisation";
        private const string PixelValueLabelNoImage = "No image under cursor.";
        DicomImage? _selectedImage;
        DicomImage? _activeImage;
        DicomImage? _magnifierImage;
        private int _activeIndex;

        private DicomImage? CurrentImage
        {
            get
            {
                if (Viewer.CurrentIndex >= 0 && Viewer.CurrentIndex < Viewer.Images.Count)
                    return Viewer.Images[Viewer.CurrentIndex];
                return null;
            }
        }

        public int CurrentIndex
        {
            get => Viewer.CurrentIndex;
            set => Viewer.CurrentIndex = value;
        }

        public DicomImageCollection Images => Viewer.Images;

        private void ContentPage_Loaded(object sender, EventArgs e)
        {
            SizeChanged += (o, args) => Viewer.Refresh();

            // Add pointer gesture recognizer
            var pointerGesture = new PointerGestureRecognizer();
            pointerGesture.PointerPressed += PointerGesture_PointerPressed;
            pointerGesture.PointerMoved += PointerGesture_PointerMoved;
            pointerGesture.PointerReleased += PointerGesture_PointerReleased;
            Viewer.GestureRecognizers.Add(pointerGesture);

#if WINDOWS
            Loaded += OnViewerLoaded;
            TryApplyWindowFixes();
#endif

            Viewer.BackColour = SKColors.Gray;
            WindowingScollZoom.IsChecked = true;
            return;
        }

#if WINDOWS
        private void TryApplyWindowFixes()
        {
            var mauiWindow = this.GetParentWindow();
            if (mauiWindow?.Handler?.PlatformView is not Microsoft.UI.Xaml.Window nativeWindow)
                return;

            var hWnd = WinRT.Interop.WindowNative.GetWindowHandle(nativeWindow);
            var windowId = Microsoft.UI.Win32Interop.GetWindowIdFromWindow(hWnd);
            var appWindow = Microsoft.UI.Windowing.AppWindow.GetFromWindowId(windowId);

            nativeWindow.DispatcherQueue.TryEnqueue(async () =>
            {
                await Task.Delay(10);

                if (appWindow == null)
                    return;

                var displayArea = DisplayArea.GetFromWindowId(windowId, DisplayAreaFallback.Primary);
                if (displayArea != null)
                {
                    // calculate resize based on users screen size
                    float widthRatio = 0.85f;
                    float heightRatio = 0.9f;
                    int targetWidth = (int)(displayArea.WorkArea.Width * widthRatio);
                    int targetHeight = (int)(displayArea.WorkArea.Height * heightRatio);
                    appWindow.Resize(new Windows.Graphics.SizeInt32(targetWidth, targetHeight));

                    // Center it
                    int x = (displayArea.WorkArea.Width - targetWidth) / 2;
                    int y = (displayArea.WorkArea.Height - targetHeight) / 2;
                    appWindow.Move(new Windows.Graphics.PointInt32(x, y));
                }
            });
        }

        public enum MouseFunctionEnums
        {
            /*  These names should match the control on the Form for GroupBoxToEnum & EnumToGroupBox utility routines to work
             *
             *  ODD values = 2D functions
             *  EVEN values = 3D functions
             */

            //  2D
            MagnifyingGlass = (1 << 2) | BitFlag2D,
            PixelValues = (2 << 2) | BitFlag2D,
            FreeRotation = (3 << 2) | BitFlag2D,
            AddAnnotation = (4 << 2) | BitFlag2D,
            EditAnnotation = (5 << 2) | BitFlag2D,

            //  3D (unsupported currently)
            RotatePanZoom = (6 << 2) | BitFlag3D,
            Crop = (7 << 2) | BitFlag3D,

            //  BOTH
            Windowing = (8 << 2) | BitFlagBoth
        }

        MouseButton _activeMouseButton = MouseButton.None;

        private enum MouseButton
        {
            None,
            Left,
            Right,
            Middle
        }

        private MouseFunctionEnums MouseFunction => Utilities.LayoutToEnum<MouseFunctionEnums>(MouseFunctionGroup);

        private void OnViewerLoaded(object? sender, EventArgs e)
        {
            if (Handler?.PlatformView is UIElement nativeElement)
            {
                nativeElement.PointerPressed += OnNativePointerPressed;
            }
        }

        private void OnNativePointerPressed(object sender, PointerRoutedEventArgs e)
        {
            var native = Handler?.PlatformView as UIElement;
            var props = e.GetCurrentPoint(native).Properties;

            if (props.IsLeftButtonPressed)
                _activeMouseButton = MouseButton.Left;
            else if (props.IsRightButtonPressed)
                _activeMouseButton = MouseButton.Right;
            else if (props.IsMiddleButtonPressed)
                _activeMouseButton = MouseButton.Middle;
            else
                _activeMouseButton = MouseButton.None;
        }
#endif


        private void OnFlipHorizontalClicked(object sender, EventArgs e)
            => ToggleFlipState(FlipState.Horizontal);

        private void OnFlipVerticalClicked(object sender, EventArgs e)
            => ToggleFlipState(FlipState.Vertical);

        private void OnRotateClockwiseClicked(object sender, EventArgs e)
            => RotateImage(RotateState.Right);

        private void OnRotateCounterClockwiseClicked(object sender, EventArgs e)
            => RotateImage(RotateState.Left);

        private void OnRestoreClicked(object sender, EventArgs e)
        {
            UpdateImage(i =>
            {
                if (GetSelectedImage() == null) return;
                GetSelectedImage().FlipState = FlipState.Normal;
                GetSelectedImage().RotateState = 0;
                GetSelectedImage().Angle = 0.0f;
                GetSelectedImage().Zoom = 1.0f;
            });
            Viewer.Refresh();
        }

        private void ToggleFlipState(FlipState targetState)
        {
            if (GetSelectedImage() == null) return;
            GetSelectedImage()!.FlipState = GetSelectedImage().FlipState switch
            {
                FlipState.Normal => targetState,
                FlipState.Horizontal when targetState == FlipState.Vertical => FlipState.Both,
                FlipState.Vertical when targetState == FlipState.Horizontal => FlipState.Both,
                FlipState.Both => targetState == FlipState.Vertical ? FlipState.Horizontal : FlipState.Vertical,
                _ => FlipState.Normal
            };

            Viewer.Refresh();
        }

        private void RotateImage(RotateState newRotateState)
        {
            if (GetSelectedImage() == null) return;
            var currentState = GetSelectedImage().RotateState;
            switch (currentState)
            {
                case RotateState.Normal:
                    GetSelectedImage().RotateState = newRotateState;
                    break;

                case RotateState.Left:
                    GetSelectedImage().RotateState =
                        newRotateState == RotateState.Left ? RotateState.Full : RotateState.Normal;
                    break;

                case RotateState.Full:
                    GetSelectedImage().RotateState =
                        newRotateState == RotateState.Left ? RotateState.Right : RotateState.Left;
                    break;

                case RotateState.Right:
                    GetSelectedImage().RotateState =
                        newRotateState == RotateState.Left ? RotateState.Normal : RotateState.Full;
                    break;
            }

            Viewer.Refresh();
        }

        private void SetSmoothingAll(bool Smooth)
        {
            // this is a simple version for the demo - other smoothing modes exist
            if (Smooth)
            {
                UpdateImages(i => i.MagnificationMode = DicomObjects.Enums.FilterMode.BSpline);
                UpdateImages(i => i.MinificationMode = DicomObjects.Enums.FilterMode.MovingAverage);
            }
            else
            {
                UpdateImages(i => i.MagnificationMode = DicomObjects.Enums.FilterMode.Replicate);
                UpdateImages(i => i.MinificationMode = DicomObjects.Enums.FilterMode.Replicate);
            }
        }

        private void SetStretchToFit(bool isStretchToFit)
        {
            UpdateImage(image =>
            {
                if (isStretchToFit)
                    image.StretchMode = StretchModes.StretchCentred;
                else
                {
                    //reset scroll & zoom to match previous stretched version
                    SKMatrix matrix = image.Matrix(new SKSize((float)Viewer.Width, (float)Viewer.Height));
                    image.Zoom = DicomGlobal.Zoom(matrix);
                    image.Scroll = new SKPoint(matrix.TransX, matrix.TransY);
                    image.StretchMode = StretchModes.NoStretch;
                }
            });
        }

        private void UpdateImages(Action<DicomImage> action)
        {
            UpdateImage(action);
        }

        public void UpdateImage(Action<DicomImage> action)
        {
            var image = GetSelectedImage();
            if (image != null)
            {
                action(GetSelectedImage());
            }
        }

        private async void OnLoadDicomClicked(object? sender, EventArgs e)
        {
            try
            {
                var fileResults = await FilePicker.PickMultipleAsync(new PickOptions
                {
                    PickerTitle = "Select DICOM files",
                    // DICOM files don’t always use the .dcm extension. May need to modify the following code in future
                    FileTypes = new FilePickerFileType(new Dictionary<DevicePlatform, IEnumerable<string>>
                    {
                        { DevicePlatform.WinUI, new[] { ".dcm" } },
                        { DevicePlatform.Android, new[] { "application/dicom" } },
                        { DevicePlatform.iOS, new[] { "public.dicom" } },
                        { DevicePlatform.MacCatalyst, new[] { ".dcm" } }
                    })
                });

                if (fileResults != null && fileResults.Any())
                {
                    //Viewer.Clear();
                    foreach (var file in fileResults)
                    {
                        var dicomImage = new DicomImage(file.FullPath);
                        Viewer.Images.Add(dicomImage);
                    }

                    SetSmoothingAll(SmoothCheck.IsChecked);
                    SetStretchToFit(StretchToFitCheck.IsChecked);
                    GetSelectedImage();
                    UpdateMarkers();
                    Viewer.AdjustMultiRowsColumns();
                }
            }
            catch (Exception ex)
            {
                await DisplayAlert("Error", $"Failed to load DICOM files: {ex.Message}", "OK");
            }
        }

        private void StretchToFit_CheckedChanged(object sender, CheckedChangedEventArgs e)
        {
            bool isChecked = e.Value;
            SetStretchToFit(isChecked);
            Viewer.Refresh();
        }

        private void SmoothImages_CheckedChanged(object sender, CheckedChangedEventArgs e)
        {
            SetSmoothingAll(SmoothCheck.IsChecked);
        }

        private void AnatomicMarkers_CheckedChanged(object sender, CheckedChangedEventArgs e)
        {
            UpdateMarkers();
            Viewer.Refresh();
        }

        private void PointerGesture_PointerPressed(object? sender, PointerEventArgs e)
        {
            Viewer.Focus();

            // clear border
            {
                UpdateImage(i => i.BorderPaint = new SKPaint
                {
                    Color = SKColors.Transparent,
                    StrokeWidth = 0
                });
            }

            // Update selected image
            var viewerPoint = e.GetPosition(Viewer); // XY coordinates relative to the viewer
            if (viewerPoint.HasValue)
            {
                SKPointI skPoint = new SKPointI((int)Math.Round(viewerPoint.Value.X * DeviceDisplay.MainDisplayInfo.Density),
                    (int)Math.Round(viewerPoint.Value.Y * DeviceDisplay.MainDisplayInfo.Density));
                MouseMoveDownPosition = skPoint;
                _activeIndex = Viewer.ImageIndex(skPoint);
                if (_activeIndex >= 0 && _activeIndex < Viewer.Images.Count)
                {
                    _activeImage = Viewer.Images[_activeIndex];
                    SetSelectedImage(_activeImage);
                    StretchToFitCheck.IsChecked = GetSelectedImage()?.StretchMode != StretchModes.NoStretch; // Reflect selected image's StretchMode in checkbox
                    Viewer.Refresh();
                }
                else
                {
                    _activeImage = null;
                    if (LabelDrawing) return;
                }

                _labelStartPoint = new PointF((float)(viewerPoint.Value.X * DeviceDisplay.MainDisplayInfo.Density),
                    (float)(viewerPoint.Value.Y * DeviceDisplay.MainDisplayInfo.Density));

                switch (MouseFunction)
                {
                    case MouseFunctionEnums.Windowing:
                        SetSmoothingAll(SmoothCheck.IsChecked);
                        break;
                    case MouseFunctionEnums.AddAnnotation:
                        // skip a frame so OnNativePointerPressed can be hit first, otherwise mouse button pressed is null
                        Dispatcher.Dispatch(() => { ViewerMouseDown_AddAnnotation(); });
                        break;
                    case MouseFunctionEnums.EditAnnotation:
                        Dispatcher.Dispatch(() => { ViewerMouseDown_EditAnnotation(); });
                        break;
                    case MouseFunctionEnums.MagnifyingGlass:
                        Dispatcher.Dispatch(() => { SetMagnifier(new SKPoint(_labelStartPoint.Value.X, _labelStartPoint.Value.Y)); });
                        break;
                    case MouseFunctionEnums.PixelValues:
                        Dispatcher.Dispatch(() => { SetPixelLabel(new SKPoint(_labelStartPoint.Value.X, _labelStartPoint.Value.Y)); });
                        break;
                }

                LastPosition = new PointF((float)(viewerPoint.Value.X * DeviceDisplay.MainDisplayInfo.Density),
                    (float)(viewerPoint.Value.Y * DeviceDisplay.MainDisplayInfo.Density));
            }
        }

        #region ViewerMouseDown functions

        private void ViewerMouseDown_AddAnnotation()
        {
            if (CurrentLabel != null)
                CurrentLabel.SelectMode = SelectMode.None;

            if (_labelStartPoint != null) CurrentLabel = _labelForm.NewLabel(new SKPoint(_labelStartPoint.Value.X, _labelStartPoint.Value.Y));
            LabelDrawing = true;
            Viewer.Labels.Add(CurrentLabel);

            if (_activeMouseButton == MouseButton.Right || _labelForm.IsAngle()) //Measure
            {
                if (FirstLineLabel == null && _labelForm.IsAngle())
                {
                    FirstLineLabel = CurrentLabel;
                }
                else
                {
                    ValueLabel = new DicomLabel()
                    {
                        ScaleMode = ScaleMode.Output,
                        ScaleFontSize = false,
                        TextBrush = new SKPaint { Color = SKColors.YellowGreen, Style = SKPaintStyle.Stroke },
                        Font = _labelForm.SelectedFont,
                        Tag = LabelTag_Info,
                    };

                    if (!_options.IsLabelTransparent)
                    {
                        var color = SKColor.Parse(_options.LabelBackColorHex); // e.g. "#FF0000"
                        ValueLabel.Brush = new SKPaint
                        {
                            Color = color,
                            Style = SKPaintStyle.Stroke
                        };
                    }

                    Viewer.Labels.Add(ValueLabel);
                }
            }
        }


        private void ViewerMouseDown_EditAnnotation()
        {
            switch (_activeMouseButton)
            {
                case MouseButton.Left:
                    if (_labelStartPoint != null)
                    {
                        var p = new SKPointI((int)_labelStartPoint.Value.X, (int)_labelStartPoint.Value.Y);
                        var labelList = Viewer.LabelHits(p, true, true).Where(i => i.Tag == null).ToList();

                        if (CurrentLabel != null)
                        {
                            CurrentLabel.SelectMode = SelectMode.None;
                            CurrentLabel.Paint.Color = _labelForm.ForeColor;
                            CurrentLabel = null;
                        }

                        if (labelList.Count > 0)
                        {
                            CurrentLabel = labelList[0];
                            CurrentLabel.Paint.Color = SKColors.Yellow;
                            CurrentLabel.SelectMode = SelectMode.Box;
                        }
                    }

                    break;
                case MouseButton.Right:
                    ResizeLabel();
                    break;
                case MouseButton.Middle:
                    LabelEdge = ContentAlignment.MiddleCenter;
                    break;
            }
        }

        private void SetMagnifier(SKPoint imagePoint)
        {
            if (_activeImage != null)
            {
                if (_magnifierImage == null)
                {
                    _magnifierImage = _activeImage.Clone(true) as DicomImage;
                    if (_magnifierImage != null)
                    {
                        _magnifierImage.StretchMode = StretchModes.NoStretch;
                        Viewer.Images.Add(_magnifierImage);
                        _magnifierImage.Zoom = DicomGlobal.Zoom(_activeImage.Matrix(Viewer)) * 2;
                        _magnifierImage.Labels.Clear();
                    }
                }

                const float magnifierSize = 0.2F;
                var magnifierPoint = new SKPoint((float)(imagePoint.X / DeviceDisplay.MainDisplayInfo.Density), (float)(imagePoint.Y / DeviceDisplay.MainDisplayInfo.Density));
                SKRect rect = SKRect.Create(magnifierPoint.X / (float)Viewer.Width, magnifierPoint.Y / (float)Viewer.Height, 0, 0);
                rect.Inflate(magnifierSize / 2, magnifierSize / 2);
                if (_magnifierImage != null)
                {
                    _magnifierImage.Area = rect;

                    // Work out how far to scroll the magnified image by doing screen ==> Image==> Screen image coordinates
                    // not simply done by scaling of offset, as anisometropic pixels can cause problems
                    var imageCoords = _activeImage.ScreenToImage(imagePoint, Viewer)!.Value;
                    var magCoords = _magnifierImage.ImageToScreen(imageCoords, Viewer)!.Value;

                    _magnifierImage.Scroll = imagePoint + new SKSize(_magnifierImage.Scroll - new SKSize(magCoords));
                }
            }
        }

        private void EndMagnifier()
        {
            if (_magnifierImage != null)
            {
                Viewer.Images.Remove(_magnifierImage);
                _magnifierImage = null;
            }
        }

        private void SetPixelLabel(SKPoint p)
        {
            if (PixelValueLabel == null)
            {
                PixelValueLabel = new DicomLabel()
                {
                    LabelType = LabelType.Text,
                    ScaleMode = ScaleMode.Output,
                    TextBrush = new SKPaint { Color = SKColors.LawnGreen },
                    Font = new SKFont(SKTypeface.Default, 22)
                };
                Viewer.Labels.Add(PixelValueLabel);
            }

            // Calculate Pixel location on current image
            if (_activeImage != null)
            {
                DicomLabel l = new DicomLabel()
                {
                    LabelType = LabelType.Rectangle,
                    ScaleMode = ScaleMode.Output,
                    Area = new SKRect(p.X, p.Y, p.X + 3, p.Y + 3)
                };

                // ROI Label
                l.Rescale(Viewer, _activeIndex, ScaleMode.Image);

                // Get Pixel Coords in active image world space
                SKPoint3? worldPos = Viewer.Images[_activeIndex].ScreenToWorld(p, Viewer);
                PixelValueLabel.Text = $"[X : {worldPos?.X ?? p.X:0.0##},  Y : {worldPos?.Y ?? p.Y:0.0##}, Z: {worldPos?.Z ?? 0:0.0##}]\nPixel Value : {l.ROIMean(Viewer.Images[_activeIndex], true):0.0###}";
            }
            else
            {
                PixelValueLabel.Text = PixelValueLabelNoImage;
            }

            PixelValueLabel.Area = new SKRect(p.X, p.Y + 30, p.X + 1000, p.Y + 130);
            Viewer.Refresh();
        }

        #endregion

        private void PointerGesture_PointerMoved(object? sender, PointerEventArgs e)
        {
            try
            {
                // This is to ensure we don't select a new active image when we are simply moving the mouse
                // over an image without clicking/dragging or anything else.
                if (!((MouseFunction == MouseFunctionEnums.AddAnnotation || MouseFunction == MouseFunctionEnums.EditAnnotation) && _labelForm.ScaleMode == ScaleMode.Output)
                    && (_activeMouseButton == MouseButton.None || _activeImage == null))
                    return;

                if (GetSelectedImage() == null || Viewer.Images.Count == 0)
                {
                    return;
                }

                var point = e.GetPosition(Viewer);
                if (!LastPosition.HasValue || !point.HasValue)
                {
                    return;
                }

                var currentPosition = new PointF((float)(point.Value.X * DeviceDisplay.MainDisplayInfo.Density), (float)(point.Value.Y * DeviceDisplay.MainDisplayInfo.Density));
                var lastPos = LastPosition.Value;
                var offset = new SizeF(currentPosition.X - lastPos.X, currentPosition.Y - lastPos.Y);
                if (WindowingScollZoom.IsChecked)
                {
                    ViewerPointerMove_Windowing(offset);
                }
                else if (AddAnnotation.IsChecked && CurrentLabel != null)
                {
                    ViewerPointerMove_AddAnnotation(currentPosition);
                }
                else if (EditAnnotation.IsChecked && CurrentLabel != null)
                {
                    ViewerPointerMove_EditAnnotation(offset);
                }
                else if (MagnifyingGlass.IsChecked && _activeImage != null)
                {
                    SetMagnifier(new SKPoint(currentPosition.X, currentPosition.Y));
                }
                else if (PixelValues.IsChecked && _activeImage != null)
                {
                    SetPixelLabel(new SKPoint(currentPosition.X, currentPosition.Y));
                }
                else if (FreeRotation.IsChecked && _activeImage != null)
                {
                    _activeImage.Angle += offset.Width;
                }

                LastPosition = currentPosition;

                var now = DateTime.UtcNow;
                if (now - _lastInvalidate >= _debounceInterval)
                {
                    _lastInvalidate = now;
                    MainThread.BeginInvokeOnMainThread(() => Viewer.InvalidateSurface());
                }
            }
            catch (Exception ex)
            {
                System.Diagnostics.Debug.WriteLine($"PointerMoved Error: {ex}");
            }

            Viewer.Refresh();
        }

        #region ViewerMouseMove functions

        private void ViewerPointerMove_Windowing(SizeF offset)
        {
            switch (_activeMouseButton)
            {
                case MouseButton.Left:
                    if (_activeImage != null)
                    {
                        _activeImage.Width += offset.Width;
                        _activeImage.Level += offset.Height;
                    }

                    var windowingLabel = GetSelectedImage().Labels.FirstOrDefault(label =>
                        (string)label.Tag == LabelTag_Info && label.Text.StartsWith("Center:"));

                    if (windowingLabel != null)
                        windowingLabel.Text = $"Center: {GetSelectedImage().Level:F0}\nWidth: {GetSelectedImage().Width:F0}";

                    break;

                case MouseButton.Middle:
                    if (_activeImage != null) _activeImage.Scroll += new SKPoint(offset.Width, offset.Height);
                    break;

                case MouseButton.Right:
                    if (StretchToFitCheck.IsChecked)
                        return;

                    if (_activeImage != null)
                    {
                        var imageRefPoint = _activeImage.ScreenToImage(MouseMoveDownPosition, Viewer);

                        if (imageRefPoint.HasValue)
                        {
                            _activeImage.Zoom *= (float)Math.Pow(1.01, offset.Width);

                            var screenLocation = _activeImage.ImageToScreen(imageRefPoint.Value, Viewer);
                            if (screenLocation != null)
                            {
                                var adjustment = new SKPoint(MouseMoveDownPosition.X - screenLocation.Value.X, MouseMoveDownPosition.Y - screenLocation.Value.Y);
                                _activeImage.Scroll += adjustment;
                            }
                        }
                    }

                    // No need to manually update zoom and the current frame index label if LabelType.Formatted is used to display them
                    break;
            }
        }


        private void ViewerPointerMove_AddAnnotation(PointF currentPosition)
        {
            if (LabelDrawing)
            {
                if (CurrentLabel != null)
                {
                    CurrentLabel.Area = new SKRect(CurrentLabel.Area.Left, CurrentLabel.Area.Top, currentPosition.X, currentPosition.Y);
                    if (CurrentLabel.LabelType == LabelType.Polygon || CurrentLabel.LabelType == LabelType.PolyLine)
                        CurrentLabel.AddPoint(new SKPoint(currentPosition.X, currentPosition.Y));

                    if (ValueLabel != null) // Measurement in progress
                    {
                        if (FirstLineLabel != null && FirstLineLabel != CurrentLabel)
                        {
                            ValueLabel.Text = $"{Math.Abs(Math.Atan2(FirstLineLabel.Height, FirstLineLabel.Width) - Math.Atan2(CurrentLabel.Height, CurrentLabel.Width)) * 180 / Math.PI:#.#}°";
                        }
                        else
                        {
                            CurrentLabel.Rescale(Viewer, _activeIndex, false, ScaleMode.Image);
                            switch (CurrentLabel.LabelType)
                            {
                                case LabelType.Line:
                                case LabelType.PolyLine:
                                    ValueLabel.Text = $"{CurrentLabel.ROILength(_activeImage):####.#} {CurrentLabel.ROIDistanceUnits(_activeImage)}";
                                    break;
                                case LabelType.Polygon:
                                case LabelType.Rectangle:
                                case LabelType.Ellipse:
                                    ValueLabel.Text = $"Area = {CurrentLabel.ROIArea(Viewer, Viewer.CurrentIndex):#.#} {CurrentLabel.ROIDistanceUnits(_activeImage)}²"
                                                      + $"\r\nMean value = {CurrentLabel.ROIMean(_activeImage, true):#.#}";
                                    break;
                                default:
                                    ValueLabel.Text = string.Empty;
                                    break;
                            }

                            CurrentLabel.Rescale(Viewer, _activeIndex, false, ScaleMode.Output);
                        }

                        ValueLabel.Area = new SKRect(currentPosition.X, currentPosition.Y - 20, currentPosition.X + 100, currentPosition.Y - 20 + 100);
                    }
                }
            }
        }

        private void ViewerPointerMove_EditAnnotation(SizeF offset)
        {
            if (CurrentLabel == null)
                return;

            if (_activeMouseButton == MouseButton.Right || _activeMouseButton == MouseButton.Middle)
            {
                int tempIndex = (_activeIndex > 0 && _activeIndex < Viewer.Images.Count) ? _activeIndex : 0;
                CurrentLabel.Adjust(LabelEdge, new SKSize(offset.Width, offset.Height), Viewer, tempIndex);
            }
        }

        #endregion


        private void PointerGesture_PointerReleased(object? sender, PointerEventArgs e)
        {
            LastPosition = null;
            _labelStartPoint = null;

            switch (MouseFunction)
            {
                case MouseFunctionEnums.Windowing:
                    SetSmoothingAll(SmoothCheck.IsChecked);
                    break;
                case MouseFunctionEnums.MagnifyingGlass:
                    EndMagnifier();
                    break;
                case MouseFunctionEnums.PixelValues:
                    Viewer.Labels.Remove(PixelValueLabel);
                    PixelValueLabel = null;
                    break;
                case MouseFunctionEnums.AddAnnotation:
                    ViewerMouseUp_AddAnnotation();
                    break;
            }

            Viewer.Refresh();
        }


        private async void ViewerMouseUp_AddAnnotation()
        {
            if (_labelForm.IsAngle())
            {
                if (FirstLineLabel == CurrentLabel)
                {
                    CurrentLabel = null;
                    return;
                }
                else
                {
                    MoveLabelViewerToImage(FirstLineLabel);
                    FirstLineLabel = null;
                }
            }

            LabelDrawing = false;

            if (_labelForm.ScaleMode == ScaleMode.Output)
            {
                Viewer.Labels.Add(CurrentLabel);
            }
            else
            {
                if (_activeImage != null)
                {
                    if (ValueLabel != null) // Measure
                    {
                        MoveLabelViewerToImage(ValueLabel);
                        ValueLabel = null;
                    }

                    MoveLabelViewerToImage(CurrentLabel);
                }
            }

            var label = CurrentLabel;
            if (label != null && label.LabelType == LabelType.Text)
            {
                if (Enum.TryParse<LabelType>(_options.LabelType, out var parsedType) && parsedType == LabelType.Text)
                {
                    string result = await DisplayPromptAsync("Enter Label Text", "");
                    if (!string.IsNullOrEmpty(result))
                    {
                        label.Text = result;
                        Viewer.Refresh();
                    }
                }

                CurrentLabel = null;
                _activeMouseButton = MouseButton.None;
            }
        }

        private void Viewer_DataChanged(object Sender)
        {
            _options.MultiColumns = Viewer.MultiColumns;
            _options.MultiRows = Viewer.MultiRows;

            // UpdateMarkers();
            UpdateInformation();

        }

        private void MoveLabelViewerToImage(DicomLabel? label)
        {
            if (label == null)
                return;

            Viewer.Labels.Remove(label);
            GetSelectedImage().Labels.Add(label);
            label.Rescale(Viewer, _activeIndex, true, _labelForm.ScaleMode);
        }

        private void UpdateInformation()
        {
            foreach (DicomImage image in Viewer.Images)
            {
                image.Labels.Remove(image.Labels.Where(label => (string)label.Tag == LabelTag_Info).ToList());
                image.Labels.Remove(image.Labels.Where(label => (string)label.Tag == BottomRightLabelTagInfo).ToList());
            }

            if (ShowPatientInfo.IsToggled)
            {
                ShowPatientInfo_Toggled(null, new ToggledEventArgs(true));
            }
        }

        private void ShowPatientInfo_Toggled(object? sender, ToggledEventArgs e)
        {
            if (ShowPatientInfo.IsToggled)
            {
                foreach (var image in Viewer.Images)
                {
                    if (image == null) continue;

                    DicomLabel windowingLabel = new DicomLabel
                    {
                        LabelType = LabelType.Text,
                        ScaleMode = ScaleMode.Cell,
                        Font = new SKFont(SKTypeface.Default, 25),
                        Text = $"Center: {image.Level}\nWidth: {image.Width}",
                        ForeColor = SKColors.AntiqueWhite,
                        Area = SKRect.Create(0, 0, 1, 1),
                        Tag = LabelTag_Info,
                        HorizontalAlign = SKTextAlign.Left,
                        VerticalAlign = VerticalAlignMode.Top
                    };

                    DicomLabel patientInfoLabel = new DicomLabel
                    {
                        LabelType = LabelType.Text,
                        ScaleMode = ScaleMode.Cell,
                        Font = new SKFont(SKTypeface.Default, 25),
                        Text = $"{image[Keyword.PatientName].Value}\n{image.PatientID}",
                        ForeColor = SKColors.AntiqueWhite,
                        Area = SKRect.Create(0, 0, 1, 1),
                        Tag = LabelTag_Info,
                        HorizontalAlign = SKTextAlign.Right,
                        VerticalAlign = VerticalAlignMode.Top
                    };

                    // label at bottom right corner - Zoom and current frame index
                    DicomLabel bottomRightLabel = new DicomLabel
                    {
                        LabelType = LabelType.Formatted,
                        ScaleMode = ScaleMode.Cell,
                        Font = new SKFont(SKTypeface.Default, 25),
                        Text = frameAndZoomLabelText,
                        ForeColor = SKColors.Purple,
                        Area = SKRect.Create(0, 0, 1, 1),
                        Tag = BottomRightLabelTagInfo,
                        HorizontalAlign = SKTextAlign.Right,
                        VerticalAlign = VerticalAlignMode.Bottom
                    };

                    image.Labels.Add(windowingLabel);
                    image.Labels.Add(patientInfoLabel);
                    image.Labels.Add(bottomRightLabel);
                }
            }
            else
            {
                foreach (var image in Viewer.Images)
                {
                    if (image == null) continue;
                    image.Labels.Remove(image.Labels.Where(label => (string)label.Tag == LabelTag_Info).ToList());
                    image.Labels.Remove(image.Labels.Where(label => (string)label.Tag == BottomRightLabelTagInfo)
                        .ToList());
                }
            }

            Viewer.Refresh();
        }

        private void UpdateMarkers()
        {
            foreach (DicomImage image in Viewer.Images)
            {
                // remove existing
                image.Labels.Remove(image.Labels.Where(item => (string)item.Tag == LabelTag_Marker).ToList());

                if (AnatomicMarkers.IsChecked)
                {
                    DicomLabel label = MarkerLabel(image);
                    label.Left = 0.05F;
                    label.Top = 0.25F;
                    label.Text = LeftText;
                    label.HorizontalAlign = SKTextAlign.Left;
                    label.VerticalAlign = VerticalAlignMode.Center;
                    label.Font = new SKFont(SKTypeface.FromFamilyName("Arial"), 22);

                    label = MarkerLabel(image);
                    label.Left = 0.25F;
                    label.Top = 0.05F;
                    label.Text = TopText;
                    label.HorizontalAlign = SKTextAlign.Center;
                    label.VerticalAlign = VerticalAlignMode.Top;
                    label.Font = new SKFont(SKTypeface.FromFamilyName("Arial"), 22);

                    label = MarkerLabel(image);
                    label.Left = 0.5F;
                    label.Top = 0.25F;
                    label.Text = RightText;
                    label.HorizontalAlign = SKTextAlign.Right;
                    label.VerticalAlign = VerticalAlignMode.Center;
                    label.Font = new SKFont(SKTypeface.FromFamilyName("Arial"), 22);

                    label = MarkerLabel(image);
                    label.Top = 0.5F;
                    label.Left = 0.25F;
                    label.Text = BottomText;
                    label.HorizontalAlign = SKTextAlign.Center;
                    label.VerticalAlign = VerticalAlignMode.Bottom;
                    label.Font = new SKFont(SKTypeface.FromFamilyName("Arial"), 22);
                }
            }
        }

        private DicomLabel MarkerLabel(DicomImage image)
        {
            DicomLabel label = new DicomLabel()
            {
                LabelType = LabelType.Special,
                ScaleMode = ScaleMode.Cell,
                ScaleFontSize = true,
                Font = _labelForm.SelectedFont,
                Area = new SKRect(0, 0, 0.45F, 0.45F),
                TextBrush = new SKPaint { Color = SKColors.Red },
                Tag = LabelTag_Marker
            };
            image.Labels.Add(label);
            return label;
        }

        private void ShowRuler_Toggled(object sender, ToggledEventArgs e)
        {
            foreach (var image in Viewer.Images)
            {
                image.Labels.Remove(image?.Labels.Where(label => (string)label.Tag == LabelTag_Ruler).ToList());

                if (e.Value)
                {
                    DicomLabel label = new DicomLabel
                    {
                        LabelType = LabelType.Ruler,
                        ScaleMode = ScaleMode.Cell,
                        Font = new SKFont(SKTypeface.Default, 20),
                        ForeColor = SKColors.Cyan,
                        Area = SKRect.Create((float)0.01, (float)0.2, (float)0.03, (float)0.6),
                        Tag = LabelTag_Ruler,
                    };
                    image?.Labels?.Add(label);
                }
            }

            Viewer.Refresh();
        }

        private void OnDeleteAllClicked(object sender, EventArgs e)
        {
            Viewer.Clear();
            SetSelectedImage(null);
        }

        private void OnDeleteSelectedImageClicked(object sender, EventArgs e)
        {
            Viewer.Images.Remove(GetSelectedImage());
            Viewer.Refresh();
            Viewer.AdjustMultiRowsColumns();
        }

        private void SetSelectedImage(DicomImage? image)
        {
            _selectedImage = image;
            if (_selectedImage != null)
            {
                UpdateImage(i =>
                {
                    i.BorderPaint = new SKPaint
                    {
                        Style = SKPaintStyle.Stroke,
                        Color = SKColors.DarkRed,
                        StrokeWidth = 3,
                        IsAntialias = true
                    };
                });
            }
        }

        public DicomImage GetSelectedImage()
        {
            if (_selectedImage == null)
            {
                SetSelectedImage(CurrentImage);
            }

            return _selectedImage!;
        }

        private void OnExitClicked(object sender, EventArgs e)
        {
            Application.Current?.Quit();
        }

        private async void OnImageInformationClicked(object sender, EventArgs e)
        {
            if (GetSelectedImage() == null)
            {
                await DisplayAlert("No Image Selected", "Please select an image before viewing its information.", "OK");
                return;
            }

            string attributesText = Utilities.DicomToText(GetSelectedImage());

            //#if WINDOWS || MACCATALYST
            var infoWindow = new Window()
            {
                Page = new ImageInformationPage(attributesText),
                Title = "Image Information",
                Width = 700,
                Height = 700
            };
            Application.Current?.OpenWindow(infoWindow);
        }

        private async void OnWriteClicked(object sender, EventArgs e)
        {
            string savePath = "";

            if (GetSelectedImage() == null)
                return;
#if WINDOWS
            try
            {
                var picker = new FileSavePicker();
                var hwnd = WindowNative.GetWindowHandle(Application.Current?.Windows[0].Handler?.PlatformView);
                InitializeWithWindow.Initialize(picker, hwnd);
                picker.FileTypeChoices.Add("DICOM file", new List<string>() { ".dcm" });

                var file = await picker.PickSaveFileAsync();
                if (file != null)
                {
                    savePath = file.Path;
                }
            }
            catch (Exception ex)
            {
                Debug.WriteLine($"Windows save failed: {ex.Message}");
            }

            // have not tested MACCATALYST yet
#elif MACCATALYST
            try
            {
                var panel = new AppKit.NSSavePanel
                {
                    Title = "Save DICOM File",
                    AllowedFileTypes = new[] { "dcm" },
                    //NameFieldStringValue = "MyImage.dcm"
                };

                var result = @await panel.BeginSheetAsync(UIKit.UIApplication.SharedApplication.KeyWindow);
                if (result == 1 && panel.Url != null)
                {
                    savePath = panel.Url.Path;
                }
            }
            catch (Exception ex)
            {
                Debug.WriteLine($"macOS save failed: {ex.Message}");
            }
#endif
            try
            {
                if (!string.IsNullOrEmpty(savePath))
                {
                    GetSelectedImage()?.Write(savePath, TransferSyntaxes.ExplicitVRLittleEndian);
                }
            }
            catch (Exception ex)
            {
                await DisplayAlert("Error", $"Failed to save DICOM files: {ex.Message}", "OK");
            }
        }

        private async void OnApplyPresentationStateClicked(object sender, EventArgs e)
        {
            var fileResults = await FilePicker.PickAsync(new PickOptions
            {
                PickerTitle = "Select Presentation State to apply",
                FileTypes = new FilePickerFileType(new Dictionary<DevicePlatform, IEnumerable<string>>
                            {
                                { DevicePlatform.WinUI, new[] { ".dcm" } },
                                { DevicePlatform.Android, new[] { "application/dicom" } },
                                { DevicePlatform.iOS, new[] { "public.dicom" } },
                                { DevicePlatform.MacCatalyst, new[] { ".dcm" } }
                            })
            });

            if (fileResults != null)
            {
                UpdateImage(i => i.PresentationState = new DicomDataSet(fileResults.FullPath));
            }
        }

        private void OnPrintDICOMClicked(object sender, EventArgs e)
        {
            if (Viewer.Images.Count <= 0) return;
            DicomPrint printer = new DicomPrint()
            {
                Colour = _options.PrintInColour,
                Format = _options.SelectedPrintFormat,
                Orientation = _options.SelectedOrientation,
                FilmSize = _options.SelectedFilmSize
            };

            printer.Open(_options.PrinterNodeName, Convert.ToInt32(_options.PrinterPort), _options.PrinterAEName, _options.PrinterClientAEName);
            printer.FilmBox.Add(Keyword.MagnificationType, _options.SelectedMagnification);

            foreach (DicomImage img in Viewer.Images)
                printer.PrintImage(img, true);
            printer.Close();
        }

        private void OnPrintWindowsClicked(object sender, EventArgs e)
        {
#if WINDOWS
            DicomImage dicomImage = GetSelectedImage();

            var skBitmap = dicomImage.Bitmap();

            // Save to PNG file
            var filePath = Path.Combine(FileSystem.CacheDirectory, "PrintImage.png");
            using (var image = SKImage.FromBitmap(skBitmap))
            using (var data = image.Encode(SKEncodedImageFormat.Png, 100))
            using (var stream = File.OpenWrite(filePath))
                data.SaveTo(stream);

            var psi = new System.Diagnostics.ProcessStartInfo
            {
                FileName = filePath,
                UseShellExecute = true,
                Verb = "print"
            };

            Process.Start(psi);
#endif
        }

        private async void OnImportFromNonDicomFileClicked(object sender, EventArgs e)
        {
            try
            {
                var fileResults = await FilePicker.PickMultipleAsync(new PickOptions
                {
                    PickerTitle = "Select non-DICOM files",
                });

                if (fileResults != null && fileResults.Any())
                {
                    Viewer.Clear();
                    foreach (var file in fileResults)
                    {
                        DicomImage image = new DicomImage();
                        image.Import(file.FullPath);
                        Viewer.Images.Add(image);
                    }

                    Viewer.AdjustMultiRowsColumns();
                }
            }
            catch (Exception ex)
            {
                string message = "";
                if (ex.Message.Contains("failed to decode") || ex.Message.Contains("Could not find stream"))
                    message = "Unsupported file formats";
                await DisplayAlert("Error", $"Failed to load non-DICOM files: {(string.IsNullOrEmpty(message)? ex.Message: message)}", "OK");
            }
        }

        private async void ExportToNonDICOMFileClicked(object sender, EventArgs e)
        {
            var selectedImage = GetSelectedImage();

            if (selectedImage == null)
            {
                await DisplayAlert("No Image Selected", "Please select an image before exporting.", "OK");
                return;
            }

            //#if WINDOWS || MACCATALYST
            var exportWindow = new Window
            {
                Page = new ExportImagePage(selectedImage),
                Title = "Export Image",
                Width = 540,
                Height = 400
            };
            Application.Current?.OpenWindow(exportWindow);

            //#else
            //  await Navigation.PushModalAsync(new ExportImagePage(selectedImage));
            //#endif
        }

        private async void VerifyCECHOClicked(object sender, EventArgs e)
        {
            int status = DicomGlobal.Echo(
                _options.StorageRemoteNode,
                _options.StorageRemotePort,
                _options.QueryCallingAE,
                _options.StorageAEName
            );

            await DisplayAlert($"C-ECHO Result", $"Verify C-ECHO returned a status of {status}", "OK");
        }

        private void QueryRetrieveCFINDCGETCMOVEClicked(object sender, EventArgs e)
        {
            // #if WINDOWS || MACCATALYST
            var options = new OptionsPage(_options);
            var retrieveWindow = new Window();

            var retrievePage = new RetrieveDialogType(options, Viewer, retrieveWindow);

            retrieveWindow.Page = retrievePage;
            retrieveWindow.Title = "Query/Retrieve";
            retrieveWindow.Width = 700;
            retrieveWindow.Height = 600;

            Application.Current?.OpenWindow(retrieveWindow);
            /*#else
                await Navigation.PushModalAsync(new QueryRetrievePage());
            #endif*/
        }

        private async void SendSelectedCSTOREClicked(object sender, EventArgs e)
        {
            if (GetSelectedImage() != null)
            {
                int status = GetSelectedImage().Send(_options.StorageRemoteNode, _options.StorageRemotePort
                    , _options.QueryCallingAE, _options.StorageAEName);
                await DisplayAlert($"C-STORE Result", $"C-STORE returned a stauts of {status}", "OK");
            }
        }

        private async void SendAllCSTOREClicked(object sender, EventArgs e)
        {
            bool hasFailures = false;

            foreach (DicomImage image in Viewer.Images)
            {
                int Status = image.Send(_options.StorageRemoteNode, _options.StorageRemotePort,
                    _options.QueryCallingAE, _options.StorageAEName);
                if (Status != 0)
                {
                    hasFailures = true;
                    await DisplayAlert($"C-Store Failure",
                        $"Sending one of the images resulted in a failure with status {Status}", "OK");
                }
            }

            if (hasFailures)
            {
                await DisplayAlert("C-Store Completed with Errors",
                    "Some images failed to send.", "OK");
            }
            else
            {
                await DisplayAlert("C-Store Success",
                    "All images have been successfully sent to the server.", "OK");
            }
        }

        private void MouseFunctionChanged(object sender, CheckedChangedEventArgs e)
        {
            if (!(sender is RadioButton button) || !button.IsChecked)
                return;

            string function = button.AutomationId;

            switch (function)
            {
                case "Windowing":
                    LeftButtonFunctionLabel.Text = "Windowing";
                    MiddleButtonFunctionLabel.Text = "Scroll";
                    RightButtonFunctionLabel.Text = "Zoom";
                    WheelButtonFunctionLabel.Text = "Select";
                    break;

                case "AddAnnotation":
                    LeftButtonFunctionLabel.Text = "Add Label";
                    MiddleButtonFunctionLabel.Text = "Add Label";
                    RightButtonFunctionLabel.Text = "Add Measurement";
                    WheelButtonFunctionLabel.Text = "Select";
                    break;

                case "EditAnnotation":
                    LeftButtonFunctionLabel.Text = "Select Label";
                    MiddleButtonFunctionLabel.Text = "Move";
                    RightButtonFunctionLabel.Text = "ReSize";
                    WheelButtonFunctionLabel.Text = "Select";
                    break;

                case "MagnifyingGlass":
                    LeftButtonFunctionLabel.Text = "Magnifier";
                    MiddleButtonFunctionLabel.Text = "Magnifier";
                    RightButtonFunctionLabel.Text = "Magnifier";
                    WheelButtonFunctionLabel.Text = "Select";
                    break;

                case "PixelValues":
                    LeftButtonFunctionLabel.Text = "Show Pixel Values";
                    MiddleButtonFunctionLabel.Text = "Show Pixel Values";
                    RightButtonFunctionLabel.Text = "Show Pixel Values";
                    WheelButtonFunctionLabel.Text = "Select";
                    break;

                case "FreeRotation":
                    LeftButtonFunctionLabel.Text = "Free Rotation";
                    MiddleButtonFunctionLabel.Text = "Free Rotation";
                    RightButtonFunctionLabel.Text = "Free Rotation";
                    WheelButtonFunctionLabel.Text = "Select";
                    break;
            }
        }

        private Window? _labelSettingWindow;

        private void AddAnnotation_Tapped(object sender, Microsoft.Maui.Controls.TappedEventArgs e)
        {
            if (_labelSettingWindow != null)
            {
                return;
            }

            _labelSettingWindow = new Window();
            var labelPage = new LabelForm(_options, _labelSettingWindow);

            _labelSettingWindow.Page = labelPage;
            _labelSettingWindow.Title = "Label Settings";
            _labelSettingWindow.Width = 550;
            _labelSettingWindow.Height = 680;


            _labelSettingWindow.Destroying += (s, args) => { _labelSettingWindow = null; };

            Application.Current?.OpenWindow(_labelSettingWindow);
        }

        private void OptionsToolClicked(object sender, EventArgs e)
        {
            var optionsWindow = new Window
            {
                Page = new OptionsPage(_options),
                Title = "Options",
                Width = 700,
                Height = 700
            };

            CurrentListenPort = _options.GeneralListenPort;

            optionsWindow.Destroying += (_, __) => { Instance?.TryUpdateListeningPort(_options.GeneralListenPort); };

            Application.Current?.OpenWindow(optionsWindow);
        }

        public int CurrentListenPort { get; set; }

        public void TryUpdateListeningPort(int newPort)
        {
            if (newPort == CurrentListenPort)
                return;

            try
            {
                _server.Unlisten(CurrentListenPort);
            }
            catch (Exception ex)
            {
                Debug.WriteLine($"Unlisten failed: {ex}");
            }

            try
            {
                _server.Listen(newPort);
                CurrentListenPort = newPort;
                Debug.WriteLine($"Now listening on port {newPort}");
            }
            catch (Exception ex)
            {
                Debug.WriteLine($"Listen failed: {ex}");
            }
        }

        private void MakeNewImageClicked(object sender, EventArgs e)
        {
            Viewer.Images.Add(Utilities.MakeNewImage());
            Viewer.AdjustMultiRowsColumns();
        }

        private async void AnonymiserToolsClicked(object sender, EventArgs e)
        {
            await DisplayAlert(AnonymiserMessageBoxTitle, AnonymiserMessageBoxText, "OK");

            try
            {
                var fileResults = await FilePicker.PickMultipleAsync(new PickOptions
                {
                    PickerTitle = "Select DICOM files to anonymise",
                    FileTypes = new FilePickerFileType(new Dictionary<DevicePlatform, IEnumerable<string>>
                    {
                        { DevicePlatform.WinUI, new[] { ".dcm" } },
                        { DevicePlatform.Android, new[] { "application/dicom" } },
                        { DevicePlatform.iOS, new[] { "public.dicom" } },
                        { DevicePlatform.MacCatalyst, new[] { ".dcm" } }
                    })
                });

                if (fileResults != null && fileResults.Any())
                {
                    var filePaths = fileResults.Select(f => f.FullPath).ToArray();
                    var anonymisedDataSets = Utilities.Anonymise(filePaths);

                    foreach (var dataSet in anonymisedDataSets)
                    {
                        var dicomImage = new DicomImage(dataSet);
                        Viewer.Images.Add(dicomImage);
                    }

                    Viewer.AdjustMultiRowsColumns();
                }
            }
            catch (Exception ex)
            {
                await DisplayAlert("Error", $"Failed to anonymise files: {ex.Message}", "OK");
            }
        }

        private void CropToLastLabelClicked(object sender, EventArgs e)
        {
            var image = GetSelectedImage();
            if (image != null)
            {
                var label = image.Labels.LastOrDefault(l => l.LabelType == LabelType.Rectangle);
                if (label != null)
                {
                    var labelArea = SKRectI.Round(label.Area);

                    DicomImage CroppedImage = GetSelectedImage().SubImage(labelArea.Location, labelArea.Size, 1.0F, 1);
                    Viewer.Images.Remove(image);
                    Viewer.Images.Add(CroppedImage);
                    Viewer.AdjustMultiRowsColumns();
                }
            }
        }

        private async void AboutExampleClicked(object sender, EventArgs e)
        {
            await DisplayAlert("About Sample viewer example of DicomObjects", "This is a simple DicomViewer example based on .NET 8 MAUI. " +
                                                                              "It demonstrates most of the functionality of DicomObjects.NET 8, including Query/Retrieve (SCU), " +
                                                                              "simple storage (SCU & SCP), printing (SCU), and image manipulation, including annotations. " +
                                                                              "In order to co-exist with the sample server, this application listens, by default, on port 1111, not 104.", "OK");
        }

        private void OnStartClicked(object sender, EventArgs e)
        {
            UpdateImage(i => { i.Frame = 1; });
        }

        private void OnReverseClicked(object sender, EventArgs e)
        {
            UpdateImage(i => i.CineMode = CineMode.Reverse);
        }

        private void OnForwardClicked(object sender, EventArgs e)
        {
            UpdateImage(i => i.CineMode = CineMode.Forward);
        }

        private void OnRepeatClicked(object sender, EventArgs e)
        {
            UpdateImage(i => i.CineMode = CineMode.Repeat);
        }

        private void OnOscillateClicked(object sender, EventArgs e)
        {
            UpdateImage(i => i.CineMode = CineMode.Oscillate);
        }

        private void OnEndClicked(object sender, EventArgs e)
        {
            UpdateImage(i => { i.Frame = i.FrameCount; });
        }

        private void OnStopClicked(object sender, EventArgs e)
        {
            UpdateImage(i => i.CineMode = CineMode.Static);
        }

        private void ResizeLabel()
        {
            if (CurrentLabel != null)
            {
                SKPoint imageRelativePosition = new SKPoint(_labelStartPoint!.Value.X, _labelStartPoint.Value.Y);
                if (CurrentLabel.ScaleMode == ScaleMode.Image)
                    imageRelativePosition = GetSelectedImage().ScreenToImage(imageRelativePosition, Viewer)!.Value;

                float x = (imageRelativePosition.X - CurrentLabel.BoundingBox.Left) / CurrentLabel.BoundingBox.Width;
                float y = (imageRelativePosition.Y - CurrentLabel.BoundingBox.Top) / CurrentLabel.BoundingBox.Height;

                //int H = (x < 0.3F) ? 0 : ((x < 0.7F) ? 1 : 2); // 0 = left, 1 = middle, 2 = right
                //int V = (y < 0.3F) ? 0 : ((y < 0.7F) ? 4 : 8); // 0 = top, 4 = middle, 8 = Bottom

                int H = 0;
                if (x < 0.3F) H = 0;
                else if (x < 0.7F) H = 1;
                else H = 2;

                int V = 0;
                if (y < 0.3F) V = 0;
                else if (y < 0.7F) V = 4;
                else V = 8;

                LabelEdge = (ContentAlignment)(1 << (H + V)); // odd, but that's how ContentAlignment is defined!
            }
        }

        private void OnPointerEntered(object sender, PointerEventArgs e)
        {
            if (sender is Border border)
                border.Stroke = Colors.SkyBlue;
        }

        private void OnPointerExited(object sender, PointerEventArgs e)
        {
            if (sender is Border border)
                border.Stroke = Colors.Transparent;
        }
    }
}