﻿using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.IO;
using System.Net.Sockets;
using System.Threading;
using DicomObjects;
using DicomObjects.Codecs;
using DicomObjects.Enums;
using DicomObjects.EventArguments;
using DicomObjects.UIDs;
using DicomObjects.Validation;
using Microsoft.Extensions.Logging;

namespace DicomServer
{
    internal class MainServer
    {
        // the object which listems for connecitons
        DicomObjects.DicomServer Server;

        // The connection to the database (re-used in many places 
        // so needs to use MultipleActiveresultsSets = true
        SqlConnection DBConnection;

        // Our own AET
        string LocalAET;

        // JPEG Compression Options
        Jpeg2000CompressionOptions opt;

        // JPIP BaseURI
        const string JpipBaseUri = "http://DicomServer.co.uk:80/Jpip";

        // For console logging
        private readonly ILogger<MainServer> _logger;

        private const string SelectAllLocalPorts = "Select * From LocalPorts";
        private const string PortNumber = "portnumber";
        private const string ListeningOnPort = "Listening on port";
        private const string SelectAllRegistry = "Select * From Registry";
        private const string RegName = "RegName";
        private const string StringTxt = "String";
        private const string Value = "Value";
        private const string LogLocation = "LogLocation";
        private const string LogLevel = "loglevel";
        private const string SelectLocalAETFromConfiguration = "Select LocalAET from Configuration";
        private const string LocalAETTxt = "LocalAET";
        private const string DicomObjectsTxt = "DicomObjects";
        private const string DicomServerStarted = "DicomServer Started";
        private const string CstoreReceived = "C-STORE Received on local port ";
        private const string Kount = "Kount";
        private const string SelectCountImageTable = "select COUNT(*) as "+Kount+" from ImageTable where InstanceUID = '";
        private const string SelectAllFieldsWhereFileName = "Select * from Fields where FILENAME=1 and ForStoring=1 order by GroupID,ElementID";
        private const string ExecuteGet = "Execute GetFilename";
        private const string FileName = "filename";
        private const string SelectAllFieldsWherePatientLevel = "Select * from Fields where PatientLevel=1 and ForStoring=1 order by GroupID,ElementID";
        private const string ExecuteInsertPatient = "Execute InsertPatient";
        private const string SelectAllFieldsWhereStudyLevel = "Select * from Fields where StudyLevel=1 and ForStoring=1 order by GroupID,ElementID";
        private const string ExecuteInsertStudy = "Execute InsertStudy";
        private const string SelectAllFromFieldsWhereSeriesLevel = "Select * from Fields where SeriesLevel=1 and ForStoring=1 order by GroupID,ElementID";
        private const string ExecuteInsertSeries = "Execute InsertSeries";
        private const string SelectAllFieldsWhereImageLevel = "Select * from Fields where ImageLevel=1 and ForStoring=1 order by GroupID,ElementID";
        private const string ExecuteInsertImage = "Execute InsertImage";
        private const string DuplicateReceivedMessage = "Duplicate received for ";
        private const string Error = "ERROR : ";
        private const string Source = "; Source : ";
        private const string CEchoReceived = "C-ECHO Received on local port ";
        private const string AssociationClosed = "Association Closed --";
        private const string ReceivedOnLocalPort = "Received on local port";
        private const string Image = "IMAGE";
        private const string InvalidQueryRoot = "Invalid Query Root : ModalityWorklist";
        private const string PatientLevelOrStudyLevel = " ( PatientLevel = 1 OR StudyLevel = 1 ) ";
        private const string LevelEqual1 = "Level = 1";
        private const string SelectAllFields = "Select * from fields WHERE ";
        private const string OrderBySortOrder = " order by SortOrder";
        private const string SortOrder = "sortorder";
        private const string FieldName = "fieldname";
        private const string GroupId = "groupid";
        private const string ElementId = "elementid";
        private const string FieldSource = "fieldsource";
        private const string FieldMatch = "FieldMatch";
        private const string UI = "UI";
        private const string ForMatching = "ForMatching";
        private const string IsName = "isName";
        private const string IsDate = "isDate";
        private const string IsTime = "isTime";
        private const string As = " AS ";
        private const string Select = "SELECT ";
        private const string From = " FROM ";
        private const string View = "View ";
        private const string OrderBy = " ORDER BY ";
        private const string ErrorInServerQueryReceived = "Error in Server_QueryReceived ";
        private const string Charset = "charset";
        private const string ErrorHandlingCFind = " Error Handling C-FIND : ";
        private const string SelectAllRemoteAetsWhere = "SELECT * from RemoteAETs where remoteaet= '";
        private const string Port = "port";
        private const string IpAddress = "IPAddress";
        private const string RemoteIpIs = "===RemoteIP is ";
        private const string CallingAetIs = "===CallingAET is ";
        private const string DestinationIs = "===Destination is ";
        private const string UnknownMoveDestination = "Unknown Move Destination: ";
        private const string EqualSigns = " ===";
        private const string SelectDistinctSopClassUIdFromImageRetrievalView = "Select DISTINCT SOPClassUID from ImageRetrievalView ";
        private const string SOPClassUID = "SOPClassUID";
        private const string SelectFilenamFromImageRetrievalView = "SELECT Filename FROM ImageRetrievalView ";
        private const string SelectAllFieldsWhereSortOrder = "Select * from fields where sortorder>0 and retrievesort=1 order by SortOrder";
        private const string AssociationRequestFrom = "Association Request from";
        private const string At = "At";
        private const string AbstractSyntax = "1.2.840.10008.1.1";
        private const string AbstractSyntax1 = "1.2.840.10008.5.1.4.1.1.";
        private const string AbstractSyntax2 = "1.2.840.10008.5.1.4.1.2.";
        private const string AbstractSyntax3 = "1.2.840.10008.5.1.4.31";
        private const string J2kExtension = ".j2k";
        private const string TransferSyntax = "1.2.840.10008.1.2.4.90";
        private const string ErrorWrittingJ2K = "Error writting J2K for ";

        public MainServer(ILogger<MainServer> logger)
        {
            _logger = logger;
        }

        public void Startup(SqlConnection connection)
        {
            // database connectino - singleton
            DBConnection = connection;

            // DicomServer object - handles all DICOM communications via the linke events
            Server = new DicomObjects.DicomServer
            {
                AllowExtendedNegotiation = true,
                DefaultStatus = 0xC000
            };
            Server.AssociationRequest += Server_AssociationRequest;
            Server.QueryReceived += Server_QueryReceived;
            Server.VerifyReceived += Server_VerifyReceived;
            Server.InstanceReceived += Server_InstanceReceived;
            Server.AssociationClosed += Server_AssociationClosed;

            DicomGlobal.Timeout = 120;

            // options for handling the JPIP protocol
            opt = new Jpeg2000CompressionOptions
            {
                Levels = 5,
                Layers = 6,
                PrecinctSize = 32,
                Rate = 1.0F,
                ForJpip = true
            };
            Jpip.JpipUriBase = JpipBaseUri;
            Utils.SetupTransferSyntaxes();

            //  == LocalPorts to listen on - as specified in the database localports table ==

            using (SqlDataReader DataReader = Utils.DoQuery(DBConnection, SelectAllLocalPorts))
            {
                while (DataReader.Read())
                {
                    int port = Convert.ToInt32(DataReader[PortNumber]);
                    Server.Listen(port, AddressFamily.InterNetwork, null);
                    Server.Listen(port, AddressFamily.InterNetworkV6, null);
                    LogText($"{ListeningOnPort} {port}");
                }
            }

            // Read "Registry" Table
            // these are overrides for unusual behaviours and also control logging
            using (SqlDataReader DataReader = Utils.DoQuery(DBConnection, SelectAllRegistry))
            {
                string loglocation = "";
                int loglevel = 0;

                while (DataReader.Read())
                {
                    string regName = DataReader[RegName].ToString();
                    if (Convert.ToBoolean(DataReader[StringTxt]))
                    {
                        DicomGlobal.SetRegString(regName, DataReader[Value].ToString());
                        if (regName.ToUpper() == LogLocation.ToUpper())
                            loglocation = DataReader[Value].ToString();
                    }
                    else
                    {
                        DicomGlobal.SetRegWord(regName, Convert.ToInt32(DataReader[Value]));
                        if (regName.ToUpper() == LogLevel.ToUpper())
                            loglevel = Convert.ToInt32(DataReader[Value]);
                    }
                }
                if (loglocation != "" && loglevel != 0)
                {
                    DicomGlobal.LogToFile(loglocation, loglevel, 60);
                }
                else // log to custom logger
                {
                    int ll = loglevel >= 8 ? loglevel : 63;

                    ILoggerFactory loggerFactory = LoggerFactory.Create(config => config.AddConsole());
                    ILogger logger = loggerFactory.CreateLogger<Program>();
                    DicomGlobal.LogToLogger(logger, ll);
                }
            }

            // Read local configuration Table
            using (SqlDataReader DataReader = Utils.DoQuery(DBConnection, SelectLocalAETFromConfiguration))
            {
                if (DataReader.HasRows)
                {
                    DataReader.Read();
                    LocalAET = DataReader[LocalAETTxt].ToString();
                }
                else
                {
                    LocalAET = DicomObjectsTxt;
                }
            }

            DicomGlobal.Log(DicomServerStarted);
        }

        public void ShutDown()
        {
            Server?.UnlistenAll();
        }

        #region DICOM Event Handlers

        // incoming C-STORE operations
        void Server_InstanceReceived(object sender, InstanceReceivedArgs e)
        {
            LogText(CstoreReceived + e.RequestAssociation.LocalPort.ToString());
            string filename = Directory.GetCurrentDirectory();
            int count = 0;
            string sql = SelectCountImageTable + e.Instance.InstanceUID + "'";

            try
            {
                // validate that the object passes at least basic DICOM rules
                e.Instance.ValidationOptions = new ValidationOptions() { CharacterSet = true, DataLength = true };
                e.Instance.Validate();

                // check if it already exists in the DB (in this sample, ignore it and send warning if we have it already)
                using (SqlDataReader DataReader = Utils.DoQuery(DBConnection, sql))
                {
                    DataReader.Read();
                    count = Convert.ToInt32(DataReader[Kount]);
                }

                if (count == 0) // images does not already exist in Database
                {
                    // this mechanism usese the "fields" table to map between DICOM and database fields
                    using (SqlDataReader DataReader = Utils.DoQuery(DBConnection, SelectAllFieldsWhereFileName))
                    using (SqlDataReader DataReader2 = Utils.DoQuery(DBConnection, Utils.MakeInsertString(e.Instance, DataReader, ExecuteGet)))
                    {
                        DataReader2.Read();
                        filename = Path.Combine(filename, Utils.MakeLegalFileName(DataReader2[FileName].ToString()));
                    }

                    // Insert PatientInfo
                    using (SqlDataReader DataReader = Utils.DoQuery(DBConnection, SelectAllFieldsWherePatientLevel))
                        Utils.DoCommand(DBConnection, Utils.MakeInsertString(e.Instance, DataReader, ExecuteInsertPatient));

                    //Insert StudyInfo
                    using (SqlDataReader DataReader = Utils.DoQuery(DBConnection, SelectAllFieldsWhereStudyLevel))
                        Utils.DoCommand(DBConnection, Utils.MakeInsertString(e.Instance, DataReader, ExecuteInsertStudy));

                    //Insert SeriesInfo
                    using (SqlDataReader DataReader = Utils.DoQuery(DBConnection, SelectAllFromFieldsWhereSeriesLevel))
                        Utils.DoCommand(DBConnection, Utils.MakeInsertString(e.Instance, DataReader, ExecuteInsertSeries));

                    // determine filename and store the object
                    Utils.MakePath(filename);

                    string outputTS = e.Instance.OriginalTS;

                    // if we're creating J2K versions for JPIP, then do that asynchronously, as it is relatively slow
                    if (outputTS != TransferSyntaxes.JPIPReferenced && outputTS != TransferSyntaxes.JPIPReferencedDeflate)
                    {
                        ThreadPool.QueueUserWorkItem(SaveImageAsync, new Tuple<string, DicomDataSet>(filename, e.Instance));
                    }

                    if (outputTS == TransferSyntaxes.JPIPReferenced || outputTS == TransferSyntaxes.JPIPReferencedDeflate)
                        outputTS = TransferSyntaxes.ExplicitVRLittleEndian;

                    e.Instance.Write(filename, outputTS);

                    //Insert InstanceInfo
                    using (SqlDataReader DataReader = Utils.DoQuery(DBConnection, SelectAllFieldsWhereImageLevel))
                        Utils.DoCommand(DBConnection, Utils.MakeInsertString(e.Instance, DataReader, ExecuteInsertImage) + ", @Filename='" + filename + "'");

                    e.Status = StatusCodes.Success;
                }
                else
                {
                    LogText(DuplicateReceivedMessage + e.Instance.InstanceUID);
                    e.Status = StatusCodes.DuplicateInstanceReceived;
                }

            }
            catch (Exception ex)
            {
                e.Status = StatusCodes.RefusedOutofresources;
                e.Errors.Add(Keyword.ErrorComment, ex.Message);
                LogText(Error + ex.Message + Source + ex.Source);
            }
        }

        // simple C-ECHO handler
        void Server_VerifyReceived(object sender, VerifyReceivedArgs e)
        {
            LogText(CEchoReceived + e.RequestAssociation.LocalPort.ToString());
            e.Status = StatusCodes.Success;
        }

        // end of association (note that this has NO significance in DICOM!)
        void Server_AssociationClosed(object Sender, AssociationClosedArgs e)
        {
            LogText($"{AssociationClosed} {e.Association.CalledAET}");
        }

        // main query handler for C-FIND, C-GET & C-MOVE
        void Server_QueryReceived(object sender, QueryReceivedArgs e)
        {
            LogText($"{e.Operation} {ReceivedOnLocalPort} {e.RequestAssociation.LocalPort}");

            // this is needed due to both IMAGE and INSTANCE beingused in DICOM for the name of the lowest level
            QueryLevel level = e.Level;
            string levelString = (level == QueryLevel.IMAGE) ? Image : level.ToString().ToUpper();

            QueryRoot root = e.Root;
            DicomDataSet request = e.RequestAssociation.Request;
            if (root == QueryRoot.PatientStudyOnly)
                root = QueryRoot.Patient;

            string fieldMatch;
            string sortorder = "";
            string fieldName, fieldSource;
            string Criteria = "";
            string Values = "";
            object Value = null;

            DicomAttribute attribute;

            // class to use when true SOP class has not been offered and negotiated
            e.RequestAssociation.CoercionSOP = SOPClasses.SecondaryCapture;

            switch (e.Operation)
            {
                case DicomOperation.C_FIND:
                    try
                    {
                        DicomDataSet resultItem;
                        DicomDataSetCollection FindResults = new DicomDataSetCollection();
                        if (e.Root == QueryRoot.ModalityWorklist)
                        {
                            e.Errors.Add(Keyword.ErrorComment, InvalidQueryRoot);
                            e.Status = StatusCodes.InvalidQueryRoot;
                            return;
                        }

                        string affectedSOPClassUID = e.Request.Command[0x0000, 0x0002].Value.ToString();
                        bool extendedNegDateTime = false;

                        foreach (DicomContext context in e.RequestAssociation.AgreedContexts)
                        {
                            if (context.AbstractSyntax == affectedSOPClassUID
                                && context.ExtendedNegotiationValues != null
                                && context.ExtendedNegotiationValues.Length > 1)
                            {
                                extendedNegDateTime = (context.ExtendedNegotiationValues[1] == 1);
                                break;
                            }
                        }

                        request.Add(Keyword.SpecificCharacterSet, "");

                        // this could be enhanced to allow hierarchical queries
                        string LevelSelector;
                        if (root == QueryRoot.Study && level == QueryLevel.STUDY)
                            LevelSelector = PatientLevelOrStudyLevel;
                        else
                            LevelSelector = " ( " + levelString + LevelEqual1+" ) ";

                        var query = SelectAllFields + LevelSelector + OrderBySortOrder;

                        using (SqlDataReader DataReader = Utils.DoQuery(DBConnection, query))
                        {
                            while (DataReader.Read())
                            {
                                if (Convert.ToInt32(DataReader[SortOrder]) != 0)
                                    sortorder += "," + DataReader[FieldName].ToString().Trim();
                                attribute = request[Convert.ToInt32(DataReader[GroupId]), Convert.ToInt32(DataReader[ElementId])];

                                if (attribute.Exists)
                                {
                                    if (((attribute.Group == 0x20 && attribute.Element == 0x1200) ||
                                         (attribute.Group == 0x20 && attribute.Element == 0x1202) ||
                                         (attribute.Group == 0x20 && attribute.Element == 0x1204))
                                       && root != QueryRoot.Patient)
                                    {
                                        continue;
                                    }

                                    fieldName = DataReader[FieldName].ToString().Trim();
                                    fieldSource = DataReader[FieldSource].ToString().Trim();
                                    fieldMatch = "";
                                    var _fieldMatch = DataReader[FieldMatch].ToString();
                                    if (_fieldMatch != "")
                                        fieldMatch = _fieldMatch;

                                    Value = attribute.Value;
                                    // First put into match list if necessary

                                    // This next procedure is for multivalued attributes, which are not generally
                                    // treated as multi-valued, but MAY be multi-valued in queries.
                                    //(This is only UIDs see CP 282)
                                    if (attribute.VR == UI && Value != null)
                                        Utils.BackSlashToArray(ref Value);

                                    if (Convert.ToBoolean(DataReader[ForMatching]) && Value != null)
                                    {
                                        if (fieldMatch != "")
                                            Utils.AddMatchCriterion(ref Criteria, fieldMatch, Value);
                                        else if (Convert.ToBoolean(DataReader[IsName]))
                                            Utils.AddNameCriterion(ref Criteria, fieldSource, attribute);
                                        else if (Convert.ToBoolean(DataReader[IsDate]))
                                        {
                                            if (extendedNegDateTime)
                                                Utils.AddDateTimeCriterion(ref Criteria, fieldSource, attribute);
                                            else
                                                Utils.AddDateCriterion(ref Criteria, fieldSource, attribute);
                                        }
                                        else if (Convert.ToBoolean(DataReader[IsTime]))
                                        {
                                            if (extendedNegDateTime)
                                                Utils.AddDateTimeCriterion(ref Criteria, fieldSource, attribute);
                                            else
                                                Utils.AddTimeCriterion(ref Criteria, fieldSource, attribute);
                                        }
                                        else
                                            Utils.AddStringCriterion(ref Criteria, fieldSource, Value, true, false);
                                    }

                                    if (Values != "")
                                        Values += ",";
                                    Values += fieldSource + As + fieldName;
                                }
                            }
                        }

                        sortorder = sortorder.Substring(1);
                        string t = Select + Values + From + levelString + View + Criteria + OrderBy + sortorder;
                        LogText("t=" + t);
                        using (SqlDataReader resultSet = Utils.DoQuery(DBConnection, t))
                        {
                            while (resultSet.Read())
                            {
                                resultItem = new DicomDataSet();
                                resultItem.Add(0x8, 0x52, levelString);
                                using (SqlDataReader DataReader = Utils.DoQuery(DBConnection, SelectAllFields + LevelSelector + OrderBySortOrder))
                                {
                                    while (DataReader.Read())
                                    {
                                        attribute = request[Convert.ToInt32(DataReader[GroupId]), Convert.ToInt32(DataReader[ElementId])];
                                        if (attribute.Exists)
                                        {
                                            if (((attribute.Group == 0x20 && attribute.Element == 0x1200) ||
                                             (attribute.Group == 0x20 && attribute.Element == 0x1202) ||
                                             (attribute.Group == 0x20 && attribute.Element == 0x1204))
                                             && root != QueryRoot.Patient)
                                            {
                                                continue;
                                            }

                                            fieldName = DataReader[FieldName].ToString().Trim();
                                            if (resultSet[fieldName] != null)
                                                Value = resultSet[fieldName];
                                            

                                            if (Value is System.DBNull)
                                            {
                                                resultItem.Add(attribute.Group, attribute.Element, "");
                                            }
                                            else
                                            {
                                                try
                                                {
                                                    resultItem.Add(attribute.Group, attribute.Element, Value);
                                                    Value = null;
                                                }
                                                catch (Exception ex1)
                                                {
                                                    LogText(ErrorInServerQueryReceived + ex1.Message);
                                                }
                                            }
                                        }
                                    }
                                }

                                // constant extras - to be included even if not reqested
                                string charset = "";
                                if (level != QueryLevel.IMAGE)
                                {
                                    if (!(resultSet[Charset] is System.DBNull))
                                        charset = resultSet[Charset].ToString() + "";
                                    if (charset != "")
                                        resultItem.Add(Keyword.SpecificCharacterSet, charset);
                                }

                                //Retrieve AET - use what the client has called this server
                                resultItem.Add(Keyword.RetrieveAETitle, e.RequestAssociation.CalledAET);

                                FindResults.Add(resultItem);
                            }
                        }
                        e.SendResponse(FindResults, 0xFF00);
                        e.Status = StatusCodes.Success;
                    }
                    catch (Exception ex)
                    {
                        e.Errors.Add(Keyword.ErrorComment, ErrorHandlingCFind + ex.Message);
                        e.Status = StatusCodes.GeneralError;
                    }
                    break;
                case DicomOperation.C_GET:
                case DicomOperation.C_MOVE:
                    try
                    {
                        DicomAssociation Connection;
                        if (e.Operation == DicomOperation.C_MOVE) //C-MOVE
                        {
                            string remoteIP = "";
                            int remotePort = 0;

                            // Look into Database for Remote C-MOVE recepient
                            using (SqlDataReader reader2 = Utils.DoQuery(DBConnection, SelectAllRemoteAetsWhere + e.Destination + "'"))
                            {
                                if (reader2.HasRows)
                                {
                                    reader2.Read();
                                    remotePort = Convert.ToInt32(reader2[Port]);
                                    remoteIP = reader2[IpAddress].ToString();
                                }
                                else // If not match found then we have to assume the SCU is the C-MOVE recipient
                                {
                                    if (e.RequestAssociation.CallingAET.ToUpper().Trim() == e.Destination.ToUpper().Trim())
                                    {
                                        LogText(RemoteIpIs + e.RequestAssociation.RemoteIP + EqualSigns);
                                        LogText(CallingAetIs + e.RequestAssociation.CallingAET + EqualSigns);
                                        LogText(DestinationIs + e.Destination + EqualSigns);

                                        remoteIP = e.RequestAssociation.RemoteIP;
                                        remotePort = (int)e.RequestAssociation.LocalPort;
                                    }
                                    else // Error - Unknown Move Destination
                                    {
                                        e.Errors.Add(Keyword.ErrorComment, UnknownMoveDestination + e.Destination);
                                        e.Status = StatusCodes.MoveDestinationUnknown;
                                        return;
                                    }
                                }
                            }
                            Connection = e.InstanceAssociation;
                            string sql = SelectDistinctSopClassUIdFromImageRetrievalView + Utils.LevelBasedSelection(level, e.Request);
                            LogText(sql);
                            using (SqlDataReader reader2 = Utils.DoQuery(DBConnection, sql))
                            {
                                while (reader2.Read())
                                {
                                    var sopClassUId = reader2[SOPClassUID].ToString();
                                    if (!string.IsNullOrEmpty(sopClassUId))
                                        Connection.RequestedContexts.Add(sopClassUId);
                                }
                                Connection.CoercionSOP = SOPClasses.SecondaryCapture;
                                Connection.RequestedContexts.Add(Connection.CoercionSOP);
                            }
                            Connection.Open(remoteIP, remotePort, LocalAET, e.Destination);
                        }
                        else
                            Connection = e.RequestAssociation;

                        string query = SelectFilenamFromImageRetrievalView;
                        query += Utils.LevelBasedSelection(level, request);
                        query += OrderBy;
                        using (SqlDataReader reader2 = Utils.DoQuery(DBConnection, SelectAllFieldsWhereSortOrder))
                        {
                            while (reader2.Read())
                            {
                                query += reader2[FieldName].ToString().Trim() + ",";
                            }
                        }
                        query = query.Substring(0, query.Length - 1);
                        using (SqlDataReader reader2 = Utils.DoQuery(DBConnection, query))
                        {
                            List<string> fn = new List<string>();

                            while (reader2.Read())
                            {
                                fn.Add(reader2[FileName].ToString());
                            }

                            Connection.SendInstances(fn.ToArray());
                        }

                        bool FailedSubOp = false;
                        foreach (DicomDataSet ds in e.RequestAssociation.ReturnedCommands)
                        {
                            if (Convert.ToInt32(ds[0x0000, 0x0900].Value) != 0)
                            {
                                FailedSubOp = true;
                                break;
                            }
                        }

                        e.Status = FailedSubOp ? StatusCodes.MoveSubOperationCompleteWithOneORMoreFailures : 0;
                    }
                    catch (Exception ex)
                    {
                        e.Errors.Add(Keyword.ErrorComment, ex.Message);
                        e.Status = StatusCodes.GeneralError;
                    }

                    break;
                default:
                    break;
            }
        }

        void Server_AssociationRequest(object sender, AssociationRequestArgs e)
        {
            bool isSOPOK = false;
            LogText($"{AssociationRequestFrom} {e.Association.CallingAET} {At} : {e.Association.RemoteIP}");

            foreach (DicomContext context in e.Contexts)
            {
                // Reject all unwanted contexts
                if (context.AbstractSyntax != AbstractSyntax
                    && !context.AbstractSyntax.StartsWith(AbstractSyntax1)
                    && !context.AbstractSyntax.StartsWith(AbstractSyntax2)
                    && !context.AbstractSyntax.Equals(AbstractSyntax3)
                    )
                    context.Reject(3);
                else
                {
                    if (context.ExtendedNegotiationValues != null && context.ExtendedNegotiationValues.Length > 0)
                    {
                        if (context.ExtendedNegotiationValues[0] == 1)//Relational
                            context.ExtendedNegotiationValues[0] = 0;

                        if (context.ExtendedNegotiationValues.Length == 3)
                        {
                            if (context.ExtendedNegotiationValues[2] == 1)//Fuzzy
                                context.ExtendedNegotiationValues[2] = 0;
                        }
                    }

                    isSOPOK = true;
                }
            }

            if (!isSOPOK) // if no DICOM context then reject the whole incoming association
                e.Reject(1, 1, 2); // 1-rejected permanent 3-service-provider-rejection(presentation related) 2- app-context-not-supported
        }

        #endregion

        #region Helpers


        void SaveImageAsync(object obj)
        {
            var tInfo = (Tuple<string, DicomDataSet>)obj;
            try
            {
                tInfo.Item2.Write(tInfo.Item1 + J2kExtension, TransferSyntax, opt);
            }
            catch (Exception ex)
            {
                LogText(ErrorWrittingJ2K + tInfo.Item1);
                LogText(ex.Message);
            }
        }

        #endregion

        #region logging

        private void LogText(string s)
        {
            DicomGlobal.Log(s);
            _logger.LogInformation(s);
        }

        #endregion
    }
}
