﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using DicomObjects;
using DicomObjects.Enums;
using DicomObjects.EventArguments;
using DicomObjects.UIDs;
using Microsoft.Extensions.Logging;

namespace DicomRouter
{
    internal class DicomRouter
    {
        private const string InstanceReceivedFromPACs = "Instance Received from PACs Server at IP";
        private const string Patient = "PATIENT";
        private const string Study = "STUDY";
        private const string Series = "SERIES";
        private const string Image = "IMAGE";
        private const string DicomVerificationReceivedFromPACs = "DICOM Verification Received from PACs Server at IP";
        private const string AssociationRequestReceivedFromPACs = "Association Request Received from PACs Server at IP";
        private const string Localhost = "LOCALHOST";
        private const string LocalIp = "127.0.0.1";
        private const string ActualIpAddress = "Actual IP Address is :";
        private const string ExpectedIpAddress = "Expected IP Address is :";
        private const string IncomingCFindRequest = "Incoming C-FIND Request Received from Client";
        private const string OutgoingCFindRequestSent = "Outgoing C-FIND Request Sent to server";
        private const string IncomingCMoveRequestReceived = "Incoming C-MOVE Request Received from Client";
        private const string IncomingCGetRequestReceived = "Incoming C-GET Request Received from Client";
        private const string OutgoingCMoveRequestSent = "Outgoing C-MOVE Request Sent to server";
        private const string OutgoingCGetRequestSent = "Outgoing C-GET Request Sent to server";
        private const string OutgoingRequestComplete = "Outgoing Request Complete";
        private const string DicomVerificationReceived = "DICOM Verification Received from Client ";
        private const string InstanceReceivedFromClient = "Instance Received from Client";
        private const string AssociationRequestReceived = "Association Request Received from";
        private const string IncomingAssociationRejected = "Incoming Association Rejected Due to Unknown Calling AET";
        private const string ListeningOnPort = "Listening on port";
        private const string ForClientCommunications = "for client communications";
        private const string ForServerCommunications = "for server communications";

        DicomServer clientServer;                    // the DicomServer Objects to handle all requests from Client
        DicomServer pacsServer;                      // the DicomServer Objects to deal with PACS

        List<DicomAssociation> connections;
       
        string RouterAET;       // Application Entity Title of this Router Application
        bool CMove;             // Boolean parameter, controls whether we use C-MOVE or C-GET to handle incoming reuqests
        bool Promiscuous;       // Boolean parameter, controls whether we accept Any CallingAET or not
        string PACS_AET;        // AET of the PACS
        int PACS_Port;          // Port number the PACS is listening 
        string PACS_IP;         // IP Address of the PACS

        public void Start()
        {
            int port_for_client = Configuration.LISTEN_PORT_CLIENT;
            string logPath = Configuration.LOG_PATH;
            int logLevel = Configuration.LOG_LEVEL;
            CMove = Configuration.CMOVE;
            Promiscuous = Configuration.Promiscuous;
            RouterAET = Configuration.ROUTER_AET;
            int port_for_server = Convert.ToInt32(Configuration.LISTEN_PORT_PACS);

            var pacs_Config = Configuration.PACS_Config;

            // FIND THE PACS SERVER CONFIG
            PACS_AET = pacs_Config.Title;
            PACS_IP = pacs_Config.Address;
            PACS_Port = Convert.ToInt32(pacs_Config.Port);

            // logToFile or logToLogger
            if (Configuration.LogToConsole)
            {
                ILoggerFactory loggerFactory = LoggerFactory.Create(config => config.AddConsole());
                ILogger logger = loggerFactory.CreateLogger<Program>();
                DicomGlobal.LogToLogger(logger, Configuration.LOG_LEVEL);
            }
            else
            {
                if (!Directory.Exists(logPath))
                    Directory.CreateDirectory(logPath);
                DicomGlobal.LogToFile(logPath, logLevel);
            }
            clientServer = new DicomServer();
            pacsServer = new DicomServer();
            connections = new List<DicomAssociation>();

            clientServer.AssociationRequest += ClientServer_AssociationRequest;
            clientServer.InstanceReceived += ClientServer_InstanceReceived;
            clientServer.VerifyReceived += ClientServer_VerifyReceived;
            clientServer.QueryReceived += ClientServer_QueryReceived;

            clientServer.DefaultStatus = 0xC000;
            clientServer.Listen(port_for_client); // listens for client communication
            Log($"{ListeningOnPort} {port_for_client} {ForClientCommunications}");

            pacsServer.DefaultStatus = 0xC000;
            pacsServer.AssociationRequest += PacsServer_AssociationRequest;
            pacsServer.VerifyReceived += PacsServer_VerifyReceived;
            pacsServer.InstanceReceived += PacsServer_InstanceReceived;
            pacsServer.Listen(port_for_server);  // listens for server communication     
            Log($"{ListeningOnPort} {port_for_server} {ForServerCommunications}");
        }

        // Fires on a new thread each time a DICOM image has been received
        void PacsServer_InstanceReceived(object Sender, InstanceReceivedArgs e)
        {
            // Get C-MOVE Responses Here, Look Up In The connections Object and Find Out 
            // Which Connection to Use to Send Images Back to Client

            Log($"{InstanceReceivedFromPACs} {e.RequestAssociation.RemoteIP}");
            int status;
            status = 0xC000;
            List<DicomAssociation> cns = FindConnection(e.Instance);   // Search the Connection
            if (cns.Count > 0)
            {
                foreach (DicomAssociation cn in cns)
                {
                    cn.SendInstances(e.Instance);
                    status = cn.LastStatus;
                }
            }
            e.Status = status;
        }

        private List<DicomAssociation> FindConnection(DicomDataSet received)
        {
            List<DicomAssociation> results = new List<DicomAssociation>();
            foreach (DicomAssociation connection in connections)
            {
                DicomDataSet request = connection.Request;
                string level = request[Keyword.QueryRetrieveLevel].Value.ToString();

                if (level == Patient && received.PatientID == request.PatientID
                    || level == Study && request.StudyUID.IndexOf(received.StudyUID) != -1
                    || level == Series && request.SeriesUID.IndexOf(received.SeriesUID) != -1
                    || level == Image && request.InstanceUID.IndexOf(received.InstanceUID) != -1)
                    results.Add(connection.RequestingQuery.InstanceAssociation);
            }
            return results; // Return All Matching Results
        }

        // Respond to DICOM C-ECHO request
        void PacsServer_VerifyReceived(object Sender, VerifyReceivedArgs e)
        {
            Log($"{DicomVerificationReceivedFromPACs} {e.RequestAssociation.RemoteIP}");
            e.Status = 0;
        }

        // Respond to Connection Request from PACS Server
        void PacsServer_AssociationRequest(object Sender, AssociationRequestArgs e)
        {
            // Only check IP address here - all communication on pacsServer should be from PACS
            Log($"{AssociationRequestReceivedFromPACs} {e.Association.RemoteIP}");
            if (PACS_IP.ToUpper() == Localhost.ToUpper())
            {
                if (e.Association.RemoteIP != LocalIp)
                    e.Reject(1, 1, 1);
            }
            else
            {
                if (e.Association.RemoteIP != PACS_IP)
                    e.Reject(1, 1, 1);
                DicomGlobal.Log($"{ActualIpAddress} {e.Association.RemoteIP} {ExpectedIpAddress} {PACS_IP}");
            }
        }

        // Respond to DICOM C-FIND/C-GET/C-MOVE request from Client
        void ClientServer_QueryReceived(object Sender, QueryReceivedArgs e)
        {
            DicomAssociation Outgoing = new DicomAssociation();
            string ClientAET;   // Client AET
            int ClientPort;     // Client Port
            string ClientIP;    // Client IP Address

            if (e.Operation == DicomOperation.C_FIND) // If C-FIND Request Received
            {
                // C-FIND passthrough is very simple
                Log($"{IncomingCFindRequest} {e.RequestAssociation.CallingAET}");
                Outgoing.RequestedContexts.Add(e.Command[Keyword.AffectedSOPClassUID].Value.ToString());

                Log($"{OutgoingCFindRequestSent} {PACS_AET}");
                Outgoing.Open(PACS_IP, PACS_Port, RouterAET, PACS_AET);
                Outgoing.Find(e.Root, e.RequestAssociation.Request);
                foreach (DicomDataSet re in Outgoing.ReturnedIdentifiers)
                    re.Add(Keyword.RetrieveAETitle, RouterAET);  // Change the Retrieve AE Title to This DicomRouter//s AET

                e.SendResponse(Outgoing.ReturnedIdentifiers, 0xFF00); // Pending status
                e.Status = Outgoing.LastStatus;
            }
            else if (e.Operation == DicomOperation.C_GET || e.Operation == DicomOperation.C_MOVE)
            {
                if (e.Operation == DicomOperation.C_MOVE)
                {
                    // If We Receive a C-MOVE, We Need Make a Reverse Connection To The Client
                    Log($"{IncomingCMoveRequestReceived} {e.RequestAssociation.CallingAET}");

                    var RemoteAET = (from myRow in Configuration.RemoteAETS
                                     where myRow.Title == e.Destination
                                     select myRow).FirstOrDefault(); ;

                    if (RemoteAET == null)
                    {
                        e.Status = 0xA801; // unknown C-MOVE destination
                        return;
                    }

                    Aet aet = RemoteAET;
                    ClientPort = aet.Port;
                    ClientIP = aet.Address;
                    ClientAET = aet.Title;
                    e.InstanceAssociation.Open(ClientIP, ClientPort, RouterAET, ClientAET);
                }
                else
                {
                    Log($"{IncomingCGetRequestReceived} {e.RequestAssociation.CallingAET}");
                }

                // Now Make The Connection To The PACS
                SetOutgoingContexts(Outgoing, e.RequestAssociation, e.Root, e.Operation);
                // Open Connection To Server
                Outgoing.Open(PACS_IP, PACS_Port, RouterAET, PACS_AET);
                if (CMove)// If We Use C-MOVE to Handle Incoming Request
                {
                    Log($"{OutgoingCMoveRequestSent} {PACS_AET}");
                    connections.Add(e.RequestAssociation); // Add Incoming Request to The Global connections for later lookup
                    Outgoing.Move(e.Root, RouterAET, e.RequestAssociation.Request); // Send C-MOVE, Ask PACS to Send Images Back to Router
                    connections.Remove(e.RequestAssociation); // Remove the Incoming From the connections once the C-MOVE has finished

                    // ***********************************************************
                    e.Status = Outgoing.LastStatus == 0xFE00 ? 0xFF00 : Outgoing.LastStatus;
                    // ***********************************************************
                }
                else // if we are using C-GET
                {
                    Log($"{OutgoingCGetRequestSent} {PACS_AET}");
                    Outgoing.Get(e.Root, e.RequestAssociation.Request); // Send C-GET, Ask PACS to Send Images Back to Router
                    e.InstanceAssociation.SendInstances(Outgoing.ReturnedInstances); // Send the Images from PACS Back to Client
                    e.Status = Outgoing.LastStatus; // Send the Final Status Back to Client
                }
            }
            Log($"{OutgoingRequestComplete}");
            Outgoing.Close();
        }

        // Respond to DICOM C-ECHO request from Client
        void ClientServer_VerifyReceived(object Sender, VerifyReceivedArgs e)
        {
            // Forward C-ECHO Request From Client Onto PACS and Return the PACS//s Status Back to Client
            Log(DicomVerificationReceived + e.RequestAssociation.CallingAET);
            e.Status = DicomGlobal.Echo(PACS_IP, PACS_Port, RouterAET, PACS_AET);
        }

        // Fires on a new Thread each time a DICOM image is received from client
        void ClientServer_InstanceReceived(object Sender, InstanceReceivedArgs e)
        {
            // Send Directly Through to PACS and Return the Status
            Log($"{InstanceReceivedFromClient} {e.RequestAssociation.CallingAET}");
            e.Status = e.Instance.Send(PACS_IP, PACS_Port, RouterAET, PACS_AET);
        }

        // Respond to DICOM Connection Request from Client
        void ClientServer_AssociationRequest(object Sender, AssociationRequestArgs e)
        {
            // Serurity Checking of IP Address and AETs
            // log the incoming Association Request
            Log($"{AssociationRequestReceived} {e.Association.CallingAET} at IP {e.Association.RemoteIP}");
            if (!Promiscuous)
            {
                var remoteAETs = from c in Configuration.RemoteAETS where c.Title.Equals(e.Association.CalledAET) select c;
                if (!remoteAETs.Any()) // If CallingAET not found in the Database then reject the incoming association
                {
                    e.Reject(1, 1, 3);
                    Log($"{IncomingAssociationRejected} {e.Association.CallingAET}");
                }
            }
        }
        private static void Log(string text)
        {
            DicomGlobal.Log(text);
        }

        private void SetOutgoingContexts(DicomAssociation Outgoing, DicomAssociation Incoming, QueryRoot Root, DicomOperation Operation)
        {
            if (CMove)  // if we are using C-MOVE, CMOVE = True
            {
                // C-GET -> C-MOVE or C-MOVE -> C-MOVE
                // Only need C-MOVE SOP Class on outgoing association
                if (Root == QueryRoot.Patient)
                    Outgoing.RequestedContexts.Add(SOPClasses.PatientRootQR_MOVE);
                else if (Root == QueryRoot.Study)
                    Outgoing.RequestedContexts.Add(SOPClasses.StudyRootQR_MOVE);
            }
            else // if we are using C-GET (CMOVE = False)
            {
                // Here the problem is to know which SOP classes to offer
                if (Operation == DicomOperation.C_GET)
                {
                    // C-GET -> C-GET
                    // This option is easy - just use what we were sent                
                    foreach (DicomContext context in Incoming.AgreedContexts)
                        Outgoing.RequestedContexts.Add(context.AbstractSyntax);
                }
                else // C-MOVE Request
                {
                    // C-MOVE -> C-GET
                    // This is the difficult one - we need to "guess" which SOP classes to use
                    if (Root == QueryRoot.Patient)
                        Outgoing.RequestedContexts.Add(SOPClasses.PatientRootQR_GET);
                    else if (Root == QueryRoot.Study)
                        Outgoing.RequestedContexts.Add(SOPClasses.StudyRootQR_GET);

                    // Replace the Following Block by a Table From the Database!!!!
                    Outgoing.RequestedContexts.Add(SOPClasses.CT);
                    Outgoing.RequestedContexts.Add(SOPClasses.SecondaryCapture);
                    Outgoing.RequestedContexts.Add(SOPClasses.MR);
                    Outgoing.RequestedContexts.Add(SOPClasses.ComputedRadiography);
                    Outgoing.RequestedContexts.Add(SOPClasses.Ultrasound);
                    Outgoing.RequestedContexts.Add(SOPClasses.NuclearMedicine);
                    Outgoing.RequestedContexts.Add(SOPClasses.UltrasoundMultiframe);
                    Outgoing.RequestedContexts.Add("1.2.840.10008.5.1.4.1.1.6"); // Retired Ultrasound, added for legacy system support
                    Outgoing.RequestedContexts.Add(SOPClasses.XRayRadiofluoroscopic);
                }
            }
        }

    }
}
