﻿using DicomObjects;
using DicomObjects.DicomWeb;
using DicomObjects.Enums;
using DicomObjects.UIDs;
using Microsoft.Extensions.Logging;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Net.Security;
using System.Security.Cryptography.X509Certificates;

namespace TestClient2
{
    class Program
    {
        protected Program() { }

        const string serverURL = "https://localhost:9090";
        const string asyncServerURL = "https://localhost:9091";
        static string studyUID, seriesUID, instanceUID;
        static void Main()
        {
            DicomGlobal.SetRegWord("ThrowExceptionOnNonSuccessHttpStatusCode", 0);

            ILoggerFactory loggerFactory = LoggerFactory.Create(config => config.AddConsole());
            ILogger logger = loggerFactory.CreateLogger<Program>();
            DicomGlobal.LogToLogger(logger, 0x3F);

            Console.WriteLine("Press any key to start Stow");
            Console.ReadKey();
            Stow(new DicomDataSet("1.dcm"));

            Console.WriteLine("Press any key to start Qido");
            Console.ReadKey();
            Qido();

            Console.WriteLine("Press any key to start Wado");
            Console.ReadKey();
            Wado(new DicomDataSet() { StudyUID = studyUID, SeriesUID = seriesUID, InstanceUID = instanceUID });

            Console.WriteLine(" ============= END OF TEST ============= ");
            Console.ReadKey();            
        }
        private static void Stow(DicomDataSet ds)
        {
            // =========== sync operations ===========
            var dwc = new StowWebClient($"{serverURL}/stow")
            {
                ServerCertificateCustomValidationCallback = ValidateWindowsCertificate
            };
            var dss = new DicomDataSetCollection() { ds };

            // 1 Store Json
            Console.WriteLine(">>> dwc.Store(dss, DicomDataEncoding.Json,  TransferSyntaxes.ExplicitVRLittleEndian)");
            var r = dwc.Store(dss, DicomDataEncoding.Json, TransferSyntaxes.ExplicitVRLittleEndian);
            GetResponse(r, dwc.Result);

            // 2 Store Native
            Console.WriteLine(">>> dwc.Store(dss, DicomDataEncoding.NativeDicom,  TransferSyntaxes.ExplicitVRLittleEndian)");
            r = dwc.Store(dss, DicomDataEncoding.NativeDicom, TransferSyntaxes.ExplicitVRLittleEndian);
            GetResponse(r, dwc.Result);

            // 3 Store 
            Console.WriteLine(">>> dwc.Store(dss, ds.OriginalTS)");
            var msg = dwc.Store(dss, ds.OriginalTS);
            Console.WriteLine(msg.ToString());

            // =========== async operations ===========
            dwc = new StowWebClient($"{asyncServerURL}/stow")
            {
                ServerCertificateCustomValidationCallback = ValidateWindowsCertificate
            };

            // 1 Store Json
            Console.WriteLine(">>> dwc.StoreAsync(dss, DicomDataEncoding.Json,  TransferSyntaxes.ExplicitVRLittleEndian)");
            r = dwc.StoreAsync(dss, DicomDataEncoding.Json, TransferSyntaxes.ExplicitVRLittleEndian).Result;
            GetResponse(r, dwc.Result);

            // 2 Store Native           
            Console.WriteLine(">>> dwc.StoreAsync(dss, DicomDataEncoding.NativeDicom,  TransferSyntaxes.ExplicitVRLittleEndian)");
            r = dwc.StoreAsync(dss, DicomDataEncoding.NativeDicom, TransferSyntaxes.ExplicitVRLittleEndian).Result;
            GetResponse(r, dwc.Result);

            // 3 Store
            Console.WriteLine(">>> dwc.StoreAsync(dss, ds.OriginalTS)");
            msg = dwc.StoreAsync(dss, ds.OriginalTS).Result;
            Console.WriteLine(msg.ToString());
        }
        public static void GetResponse(DicomDataSet ds, HttpResponseMessage resp = null)
        {
            if (ds != null && ds.Count > 0)
            {
                if (ds[Keyword.ReferencedSOPSequence].Value is DicomDataSetCollection sq)
                {
                    foreach (var s in sq)
                    {
                        Console.WriteLine($"STOW Resp:  {s[Keyword.ReferencedSOPInstanceUID]}");
                    }
                }
            }
            if (resp != null)
            {
                Console.WriteLine($"{resp}");
            }
        }        
        public static bool ValidateWindowsCertificate(HttpRequestMessage msg, X509Certificate2 certificate, X509Chain chain, SslPolicyErrors sslPolicyErrors)
        {
            // check sslPolicyErrors to validate server certificate errors
            // or simply return true to fully trust server certificate
            return true;
        }
        private static void Qido()
        {
            // =========== sync operations ===========
            var dwc = new QidoWebClient($"{serverURL}/Qido")
            {
                ServerCertificateCustomValidationCallback = ValidateWindowsCertificate
            };
            var q = new DicomQuery
            {
                Level = QueryLevel.STUDY,
                Root = QueryRoot.Study
            };

            DicomDataSet queryDataSet = q.QueryDataSet();
            queryDataSet.Add(Keyword.PatientID, ""); // ds.PatientID);

            var result = dwc.Query(queryDataSet);
            Dump(">>> QIDO Query", result, dwc.Result);

            // =========== async operations ===========
            dwc = new QidoWebClient($"{asyncServerURL}/Qido")
            {
                ServerCertificateCustomValidationCallback = ValidateWindowsCertificate
            };
            result = dwc.QueryAsync(queryDataSet).Result;
            Dump(">>> QIDO QueryAsync", result, dwc.Result);

            studyUID = result.FirstOrDefault().StudyUID;

            // now try to get the series instance UID
            queryDataSet.StudyUID = studyUID;
            q.StudyUID = studyUID;
            q.Level = QueryLevel.SERIES;
            result = dwc.QueryAsync(q.QueryDataSet()).Result;
            seriesUID = result.FirstOrDefault().SeriesUID;

            // now try to get the instance UID
            q.StudyUID = studyUID;
            q.SeriesUID = seriesUID;
            q.Level = QueryLevel.INSTANCE;
            result = dwc.QueryAsync(q.QueryDataSet()).Result;
            instanceUID = result.FirstOrDefault().InstanceUID;
        }      

        private static void Wado(DicomDataSet ds)
        {
            string studyUID = ds.StudyUID;
            string seriesUID = ds.SeriesUID;
            string instanceUID = ds.InstanceUID;

            // =========== sync operations ===========
            var dwc = new WadoWebClient($"{serverURL}/Wado")
            {
                ServerCertificateCustomValidationCallback = ValidateWindowsCertificate
            };

            // retrieve native DICOM
            var result = dwc.RetrieveNative(studyUID, seriesUID, instanceUID);
            Dump(">>> WADO Retrieve Native", result, dwc.Result);

            // retrive native DICOM with only metadata (no blob)
            result = dwc.RetrieveMetaData(studyUID, seriesUID, instanceUID);
            Dump(">>> WADO Retrieve MetaData", result, dwc.Result);

            // retrieve rendered (image/jpeg, text/xml etc)
            var results = dwc.RetrieveRendered(studyUID, seriesUID, instanceUID);
            Console.WriteLine($">>> WADO Retrieve Rendered");
            Console.WriteLine($"Returned {results.Count} results:");
            foreach (var r in results)
            {
                if (r.Key.ContentType.MediaType.Contains("text"))
                    Console.WriteLine($"    MediaType={r.Key.ContentType.MediaType}. Msg={System.Text.Encoding.UTF8.GetString(r.Value)}");
                else
                    Console.WriteLine($"    MediaType={r.Key.ContentType.MediaType}");
            }
            Console.WriteLine(dwc.Result.ToString());

            // =========== async operations ===========
            dwc = new WadoWebClient($"{asyncServerURL}/Wado")
            {
                ServerCertificateCustomValidationCallback = ValidateWindowsCertificate
            };

            // retrieve native DICOM    
            result = dwc.RetrieveNativeAsync(studyUID, seriesUID, instanceUID).Result;
            Dump(">>> WADO Retrieve Native Async", result, dwc.Result);

            // retrive native DICOM with only metadata (no blob)           
            result = dwc.RetrieveMetaDataAsync(studyUID, seriesUID, instanceUID).Result;
            Dump(">>> WADO Retrieve MetaData Async", result, dwc.Result);

            // retrieve rendered (image/jpeg, text/xml etc) 
            results = dwc.RetrieveRenderedAsync(studyUID, seriesUID, instanceUID).Result;
            Console.WriteLine($">>> WADO Retrieve Rendered Async");
            Console.WriteLine($"Returned {results.Count} results:");
            foreach (var r in results)
            {
                if (r.Key.ContentType.MediaType.Contains("text"))
                    Console.WriteLine($"    MediaType={r.Key.ContentType.MediaType}. Msg={System.Text.Encoding.UTF8.GetString(r.Value)}");
                else
                    Console.WriteLine($"    MediaType={r.Key.ContentType.MediaType}");
            }
            Console.WriteLine(dwc.Result.ToString());
        }

        private static void Dump(string methodName, IEnumerable<DicomDataSet> results, HttpResponseMessage response)
        {
            Console.WriteLine($"============={methodName}=============");
            int count = 1;
            foreach (var ds in results)
            {
                Console.WriteLine($"================returned result #{count}===================");
                foreach (var at in ds)
                {
                    if (at.KeywordCode == DicomObjects.Enums.Keyword.PixelData)
                    {
                        Console.WriteLine($"({at.Group:X4}, {at.Element:X4}) : {at.Description}");
                    }
                    else
                    {
                        Console.WriteLine($"({at.Group:X4}, {at.Element:X4}) : {at.Description} : {at.Value}");
                    }
                }
                count++;
            }
            if (response == null ||
                response.StatusCode == HttpStatusCode.OK ||
                response.StatusCode == HttpStatusCode.PartialContent)
                Console.WriteLine(response.ToString());
            else
                Console.WriteLine(response.Content.ReadAsStringAsync().Result);
            Console.WriteLine($"================End of {methodName}==================");
        }
    }
}
