﻿using System;
using System.Threading;
using DicomObjects;
using DicomObjects.Enums;
using DicomObjects.EventArguments;
using DicomObjects.UIDs;
using Microsoft.Extensions.Logging;

namespace SCU
{
    class Program
    {
		private static Timer tmr_dicom_connection;

		private static DicomAssociation cn;

		private const string PressAnyKeyToSendStorageCommitment = "Press any key to send Storage Commitment to SCP";
		private const string CloseOutgoingAssociationLogMessage = "Outgoing Association Closed Normally";
		private const string PressEnterToExit = "Press Enter to exit";

		private const string Node = "localhost";
		private const string CallingAE = "SC_SCU";
		private const string CalledAE = "SC_SCP";
		private const int Port = 104;
		private const string RequestForImage = "Request for Image";
		private const string CommitmentRequest = "added for storage commitment request.";
		private const string SendingStorageCommitmentRequest = "Sending storage commitment request - Transaction UID =";
		private const string IncomingAssocationRequest = "Incoming Assocaiton Request from";
		private const string CommandFieldErrorMessage = "Fatal Error";
		private const string InstanceUIdNotCorrectErrorMessage = "AffectedSOPInstanceUID not Correct";
		private const string ResponseReceivedOnNewAssociationMessage = "Response Received on New Association";
		private const string ResponseReceivedOnOriginalAssociationMessage = "Response Received on Original Association";
		private const string ResponseReceivedOnDifferentAssociationMessage = "Response Received on Different Association";
		private const string StorageCommitmentSucceededMessage = "Storage commitment succeeded for Transaction UID:";
		private const string SucceededImageMessage = "Succeeded Image Instance UID";
		private const string StorageCommitmentFailedMessage = "Storage commitment failed for Transaction UID:";
		private const string FailedImageMessage = "Failed Image Instance UID";
		private const string WrongEventIdReceivedMessage = "Error - Wrong EventID Received";

		static void Main(string[] args)
		{
			DicomServer Server = new DicomServer();
			Server.AssociationRequest += Server_AssociationRequest;
			Server.N_Event_Report += Server_N_Event_Report;
			Server.DefaultStatus = 0x110;
			Server.Listen(105);
			tmr_dicom_connection = new Timer(new TimerCallback(Tmr_dicom_connection_Tick), null, 4000, 4000);

			Console.WriteLine(PressAnyKeyToSendStorageCommitment);
			Console.WriteLine(PressEnterToExit);

			ILoggerFactory loggerFactory = LoggerFactory.Create(config => config.AddConsole());
			ILogger logger = loggerFactory.CreateLogger<Program>();
			DicomGlobal.LogToLogger(logger, 0x3F);

			do
			{
				var key = Console.ReadKey().Key;
				if (key == ConsoleKey.Enter)
					return;

				SendSCPush();
			}
			while (true);
		}

		private static void CloseOutgoingAssociation()
		{
			tmr_dicom_connection.Change(int.MaxValue, int.MaxValue);
			if (cn != null)
			{
				cn.Close();
			}
			cn = null;
			Log(CloseOutgoingAssociationLogMessage);
		}

		private static void Tmr_dicom_connection_Tick(Object obj)
		{
			CloseOutgoingAssociation();
		}

		private static void Log(string s)
		{
			DicomGlobal.Log(s);
		}

		private static void SendSCPush()
		{
			/* STORAGE COMMITMENT REQUEST
			 
			Action Type/Name		Action Type ID		Attribute		Tag
			
			Request						
			Storage						   1		  TransactionUID (0008,1195)
			Commitment
													  Storage Media
                                                       File-Set ID   (0088,0130)
														
													  Storage Media
													   File-Set UID  (0088,0140)

													   Referenced
													       SOP
													    Sequence     (0008,1199)

													    >Referenced
													     SOP Class
													     UID           >(0008,1150)
														>Referenced
														SOP Instance
														UID			   >(0008,1155)
			 
			 */

			string[] offerTS = new string[2];
			offerTS[0] = TransferSyntaxes.ImplicitVRLittleEndian;
			offerTS[1] = TransferSyntaxes.ExplicitVRLittleEndian;

			// close possible previously opened association and create a new one
			CloseOutgoingAssociation();

			// Enable timer to close outgoing association after a time-out
			tmr_dicom_connection.Change(4000, 4000);
			cn = new DicomAssociation
			{
				// Negotiate a suitable Association with Correct SOP Class
				MetaSOPClass = SOPClasses.StorageCommitmentPush
			};
			cn.RequestedContexts.Add(SOPClasses.StorageCommitmentPush);
			cn.RequestedContexts[1].RequestorSCURole = true;
			cn.RequestedContexts[1].RequestorSCPRole = false;
			cn.RequestedContexts[1].OfferedTS = offerTS;

			cn.Open(Node, Port, CallingAE, CalledAE);

			DicomDataSet ds = new DicomDataSet();
			DicomDataSet referencedImage1 = new DicomDataSet();
			DicomDataSet referencedImage2 = new DicomDataSet();
			DicomDataSetCollection referencedImagesSequence = new DicomDataSetCollection();

			string instanceUID;

			// the next 3 lines could be repeated N times to send a request for N Images
			referencedImage1.Add(Keyword.ReferencedSOPClassUID, SOPClasses.SecondaryCapture);
			instanceUID = DicomGlobal.NewUID();
			referencedImage1.Add(Keyword.ReferencedSOPInstanceUID, instanceUID);
			referencedImagesSequence.Add(referencedImage1);
			Log($"{RequestForImage} {instanceUID} {CommitmentRequest}");

			// Add another image here
			referencedImage2.Add(Keyword.ReferencedSOPClassUID, SOPClasses.SecondaryCapture);
			instanceUID = DicomGlobal.NewUID();
			referencedImage2.Add(Keyword.ReferencedSOPInstanceUID, instanceUID);
			referencedImagesSequence.Add(referencedImage2);
			Log($"{RequestForImage} {instanceUID} {CommitmentRequest}");

			ds.Add(Keyword.ReferencedSOPSequence, referencedImagesSequence);
			ds.Add(Keyword.TransactionUID, DicomGlobal.NewUID());

			Log($"{SendingStorageCommitmentRequest} {ds[Keyword.TransactionUID].Value}");

			// Send Storage Commitment Push
			cn.NAction(SOPClasses.StorageCommitmentPush, WellKnownInstances.StorageCommitmentPushModelInstance, 1, ds);
		}

		private static void Server_AssociationRequest(object sender, AssociationRequestArgs e)
		{
			Log($"{IncomingAssocationRequest} {e.Association.CallingAET} at {e.Association.RemoteIP}");
		}

		private static void Server_N_Event_Report(object sender, N_Event_ReportArgs e)
		{
			string AffectedSOPInstanceUID = Convert.ToString(e.Command[Keyword.AffectedSOPInstanceUID].Value);
			int CommandField = Convert.ToInt32(e.Command[Keyword.CommandField].Value.ToString());
			int status = 0x110;

			if (CommandField != 0x100)
			{
				Log(CommandFieldErrorMessage);
			}

			if (AffectedSOPInstanceUID != WellKnownInstances.StorageCommitmentPushModelInstance)
			{
				status = 0x112; // no such object instance
				Log(InstanceUIdNotCorrectErrorMessage);
			}

			if (cn == null)
			{
				Log(ResponseReceivedOnNewAssociationMessage);
			}
			else
			{
				Log(cn.AssociationNumber == e.RequestAssociation.AssociationNumber
						? ResponseReceivedOnOriginalAssociationMessage
						: ResponseReceivedOnDifferentAssociationMessage);
			}

			switch (e.EventID)
			{
				case 1: // list all successful requests
					Log($"{StorageCommitmentSucceededMessage} {e.Request[Keyword.TransactionUID].Value}");
					DicomDataSetCollection dss = (DicomDataSetCollection)e.Request[Keyword.ReferencedSOPSequence].Value;
					foreach (DicomDataSet ds in dss)
					{
						Log($"{SucceededImageMessage} {ds[Keyword.ReferencedSOPInstanceUID].Value}");
					}
					break;
				case 2: // list all successful requests as well as failed requests
					Log($"{StorageCommitmentFailedMessage} {e.Request[Keyword.TransactionUID].Value}");
					DicomDataSetCollection dss1 = (DicomDataSetCollection)e.Request[Keyword.ReferencedSOPSequence].Value;
					foreach (DicomDataSet ds in dss1)
					{
						Log($"{SucceededImageMessage} {ds[Keyword.ReferencedSOPInstanceUID].Value}");
					}
					DicomDataSetCollection dss2 = (DicomDataSetCollection)e.Request[Keyword.FailedSOPSequence].Value;
					foreach (DicomDataSet ds in dss2)
					{
						Log($"{FailedImageMessage} {ds[Keyword.ReferencedSOPInstanceUID].Value}");
					}
					break;
				default:
					status = 0x123; // no such action type
					Log(WrongEventIdReceivedMessage);
					break;
			}
			e.Status = status;
		}
	}
}
