﻿using DicomObjects;
using DicomObjects.Enums;
using Microsoft.Extensions.Logging;
using System;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Runtime.InteropServices;

namespace SCP
{
    class Program
    {
        #region "Closing event halder"
        [DllImport("Kernel32")]
        private static extern bool SetConsoleCtrlHandler(EventHandler handler, bool add);

        private delegate bool EventHandler(CtrlType sig);
        static EventHandler _handler;
        #endregion

        #region Constants
        const bool DISABLE_DATE_MATCHING = false;
        const bool ENABLE_LOGGING = true;
        const string CONNECTION_STRING = @"Server=.\SQLEXPRESS;Database=MWLDB;Trusted_Connection=True;";
        #endregion


        #region "Variables"

        private static DicomServer MWL_Server;
        private static int port = 110;

        #endregion
        static void Main(string[] args)
        {
            _handler += new EventHandler(Handler);
            SetConsoleCtrlHandler(_handler, true);

            //Logging
            if (ENABLE_LOGGING)
            {
                ILoggerFactory loggerFactory = LoggerFactory.Create(config => config.AddConsole());
                ILogger logger = loggerFactory.CreateLogger<Program>();
                DicomGlobal.LogToLogger(logger, 0x3F);
            }

            MWL_Server = new DicomServer();
            MWL_Server.VerifyReceived += MWL_Server_VerifyReceived; ;
            MWL_Server.AssociationRequest += MWL_Server_AssociationRequest; ;
            MWL_Server.QueryReceived += MWL_Server_QueryReceived; ;
            MWL_Server.Listen(port);
            Console.WriteLine("MWL Server started. Listening on Port: " + port);

            Console.ReadKey();
        }

        private static bool Handler(CtrlType sig)
        {
            switch (sig)
            {
                case CtrlType.CTRL_C_EVENT:
                    MWL_Server.UnlistenAll();
                    Console.WriteLine("MWL Server Stopped! ");
                    return false;
                case CtrlType.CTRL_LOGOFF_EVENT:
                    MWL_Server.UnlistenAll();
                    Console.WriteLine("MWL Server Stopped! ");
                    return false;
                case CtrlType.CTRL_SHUTDOWN_EVENT:
                    MWL_Server.UnlistenAll();
                    Console.WriteLine("MWL Server Stopped! ");
                    return false;
                case CtrlType.CTRL_CLOSE_EVENT:
                    MWL_Server.UnlistenAll();
                    Console.WriteLine("MWL Server Stopped! ");
                    return false;
                default:
                    return false;
            }
        }

        private static void MWL_Server_QueryReceived(object Sender, DicomObjects.EventArguments.QueryReceivedArgs e)
        {
            DicomDataSetCollection NullSequence = new DicomDataSetCollection();
            DicomDataSet rq;
            DicomDataSet rq1 = null;
            int status = 0;
            List<ExamsScheduled> results = new List<ExamsScheduled>();

            try
            {
                using (SqlConnection connection = new SqlConnection(CONNECTION_STRING))
                {
                    connection.Open();
                    if (e.Root != QueryRoot.ModalityWorklist)
                    {
                        e.Errors.Add(Keyword.ErrorComment, "Invalid Query Root : None");
                        e.Status = StatusCodes.InvalidQueryRoot;
                        return;
                    }
                    // Get the Imcoming Query Request
                    rq = e.RequestAssociation.Request;

                    // In a "Real" MWl server, this would either itself be a complex linked query, or a reference to such
                    // a query as a "view" or similar in the underlying database

                    //  Using SQL

                    // the "where 1=1" makes the syntax of adding further conditions simpler, as all are then " AND x=y"
                    string sql = "SELECT * from ExamsScheduled Where 1=1";
                    Utils_SQL.AddCondition(ref sql, rq[Keyword.PatientID], "PatientID");
                    Utils_SQL.AddNameCondition(ref sql, rq[Keyword.PatientName], "surname", "Forename");

                    if (rq[Keyword.ScheduledProcedureStepSequence].ExistsWithValue)
                    {
                        DicomDataSetCollection rqs;

                        rqs = (DicomDataSetCollection)rq[Keyword.ScheduledProcedureStepSequence].Value;
                        rq1 = rqs[0];

                        // Required Matching keys
                        Utils_SQL.AddCondition(ref sql, rq1[Keyword.ScheduledStationAETitle], "ScheduledAET");
                        Utils_SQL.AddCondition(ref sql, rq1[Keyword.PerformingPhysicianName], "PerformingPhysician");
                        Utils_SQL.AddCondition(ref sql, rq1[Keyword.Modality], "Modality");

                        // if only date is specified, then using standard matching
                        //but if both are specified, then MWL defines a combined match

                        if (DISABLE_DATE_MATCHING)
                        {
                            if (rq1[Keyword.ScheduledProcedureStepStartDate].ExistsWithValue && rq1[Keyword.ScheduledProcedureStepStartTime].ExistsWithValue) // if both Date and Time are specified
                            {
                                Utils_SQL.AddDateTimeCondition(ref sql, rq1[Keyword.ScheduledProcedureStepStartDate], rq1[Keyword.ScheduledProcedureStepStartTime], "ExamDateAndTime");
                            }
                            else if (rq1[Keyword.ScheduledProcedureStepStartDate].ExistsWithValue) // if Date is specified
                            {
                                Utils_SQL.AddDateCondition(ref sql, rq1[Keyword.ScheduledProcedureStepStartDate], "ExamDateAndTime");
                            }
                        }

                        // Optional (but commonly used) matching keys.
                        Utils_SQL.AddCondition(ref sql, rq1[Keyword.ScheduledProcedureStepLocation], "ExamRoom");
                        Utils_SQL.AddCondition(ref sql, rq1[Keyword.ScheduledProcedureStepDescription], "ExamDescription");
                    }

                    sql = sql + " Order by Surname, Forename";
                    Console.WriteLine("SQL query statement:" + Environment.NewLine + sql);

                    using (SqlCommand command = new SqlCommand(sql, connection))
                    {
                        using (SqlDataReader reader = command.ExecuteReader())
                        {
                            while (reader.Read())
                            {
                                results.Add(new ExamsScheduled()
                                {
                                    AccessionNumber = reader["AccessionNumber"].ToString(),
                                    DateOfBirth = Convert.ToDateTime(reader["DateOfBirth"]),
                                    ExamDateAndTime = Convert.ToDateTime(reader["ExamDateAndTime"]),
                                    ExamDescription = reader["ExamDescription"].ToString(),
                                    ExamRoom = reader["ExamRoom"].ToString(),
                                    Forename = reader["Forename"].ToString(),
                                    HospitalName = reader["HospitalName"].ToString(),
                                    Modality = reader["Modality"].ToString(),
                                    PatientID = reader["PatientID"].ToString(),
                                    PerformingPhysician = reader["PerformingPhysician"].ToString(),
                                    ProcedureID = reader["ProcedureID"].ToString(),
                                    ProcedureStepID = reader["ProcedureStepID"].ToString(),
                                    ReferringPhysician = reader["ReferringPhysician"].ToString(),
                                    ScheduledAET = reader["ScheduledAET"].ToString(),
                                    Sex = reader["Sex"].ToString(),
                                    StudyUID = reader["StudyUID"].ToString(),
                                    Surname = reader["Surname"].ToString(),
                                    Title = reader["Title"].ToString()
                                });
                            }
                        }
                    }

                    //  Parsing result 
                    DicomDataSet rr1;
                    DicomDataSet rr;
                    DicomDataSetCollection rrs;
                    foreach (ExamsScheduled result in results)
                    {
                        rr1 = new DicomDataSet();
                        rr = new DicomDataSet();
                        rrs = new DicomDataSetCollection { rr1 };

                        if (rq1 != null)
                        {
                            rr.Add(Keyword.ScheduledProcedureStepSequence, rrs);
                        }

                        // add results to  "main" dataset

                        AddResultItem(rr, rq, Keyword.AccessionNumber, result.AccessionNumber);    // T2
                        AddResultItem(rr, rq, Keyword.InstitutionName, result.HospitalName);
                        AddResultItem(rr, rq, Keyword.ReferringPhysicianName, result.ReferringPhysician); // T2

                        AddResultItem(rr, rq, Keyword.PatientName, result.Surname + "^" + result.Forename + "^^" + result.Title); //T1
                        AddResultItem(rr, rq, Keyword.PatientID, result.PatientID); // T1
                        AddResultItem(rr, rq, Keyword.PatientBirthDate, result.DateOfBirth); // T2
                        AddResultItem(rr, rq, Keyword.PatientSex, result.Sex); //T2

                        AddResultItem(rr, rq, Keyword.StudyInstanceUID, result.StudyUID); // T1

                        AddResultItem(rr, rq, Keyword.RequestingPhysician, result.ReferringPhysician); //T2
                        AddResultItem(rr, rq, Keyword.RequestedProcedureDescription, result.ExamDescription); //T1C

                        AddResultItem(rr, rq, Keyword.RequestedProcedureID, result.ProcedureID); // T1

                        // Scheduled Procedure Step sequence T1
                        // add results to procedure step dataset
                        // Return if requested
                        if (rq1 != null)
                        {
                            AddResultItem(rr1, rq1, Keyword.ScheduledStationAETitle, result.ScheduledAET); // T1
                            AddResultItem(rr1, rq1, Keyword.ScheduledProcedureStepStartDate, result.ExamDateAndTime); //T1
                            AddResultItem(rr1, rq1, Keyword.ScheduledProcedureStepStartTime, result.ExamDateAndTime); //T1
                            AddResultItem(rr1, rq1, Keyword.Modality, result.Modality); // T1

                            AddResultItem(rr1, rq1, Keyword.ScheduledPerformingPhysicianName, result.PerformingPhysician); //T2
                            AddResultItem(rr1, rq1, Keyword.ScheduledProcedureStepDescription, result.ExamDescription); // T1C
                            AddResultItem(rr1, rq1, Keyword.ScheduledProcedureStepID, result.ProcedureStepID); // T1
                            AddResultItem(rr1, rq1, Keyword.ScheduledStationName, result.ExamRoom); //T2
                            AddResultItem(rr1, rq1, Keyword.ScheduledProcedureStepLocation, result.ExamRoom); //T2
                        }

                        // Put blanks in for unsupported fields which are type 2 (i.e. must have a value even if NULL)
                        // In a real server, you may wish to support some or all of these, but they are not commonly supported

                        AddResultItem(rr, rq, Keyword.ReferencedStudySequence, NullSequence);
                        AddResultItem(rr, rq, Keyword.Priority, "");
                        AddResultItem(rr, rq, Keyword.PatientTransportArrangements, "");
                        AddResultItem(rr, rq, Keyword.AdmissionID, "");
                        AddResultItem(rr, rq, Keyword.CurrentPatientLocation, "");
                        AddResultItem(rr, rq, Keyword.ReferencedPatientSequence, NullSequence);
                        AddResultItem(rr, rq, Keyword.PatientWeight, "");
                        AddResultItem(rr, rq, Keyword.ConfidentialityConstraintOnPatientDataDescription, "");
                        Console.WriteLine("Sending response ..");
                        // Send Reponse Back
                        e.SendResponse(rr, StatusCodes.Pending);
                    }
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine("Error executing Database -- " + ex.Message);
                status = StatusCodes.GeneralError; // Error, unable to process!
            }
            finally
            {
                Console.WriteLine("Database Disconnected Normally");
            }
            e.Status = status;
        }

        private static void MWL_Server_AssociationRequest(object Sender, DicomObjects.EventArguments.AssociationRequestArgs e)
        {
            Console.WriteLine("Association Request from " + e.Association.CallingAET + " At " + e.Association.RemoteIP);
            // Reject any irrelevant Contexts			
            foreach (DicomContext context in e.Contexts)
            {
                if (context.AbstractSyntax != DicomObjects.UIDs.SOPClasses.ModalityWorklistQRFIND
                    && context.AbstractSyntax != DicomObjects.UIDs.SOPClasses.Verification)
                {
                    context.Reject(3); // 3- Abstract Syntax Not Supported
                }
            }
        }

        private static void MWL_Server_VerifyReceived(object Sender, DicomObjects.EventArguments.VerifyReceivedArgs e)
        {
            e.Status = 0;
        }

        internal static void AddResultItem(DicomDataSet DataSet, DicomDataSet request, Keyword keyword, object v)
        {
            // Only send items which have been requested
            if (request[keyword].Exists)
            {
                if (v == null)
                    v = "";
                DataSet.Add(keyword, v);
            }
        }
    }

    internal static class StatusCodes
    {
        public const int Success = 0x0;
        public const int GeneralError = 0xC000;
        public const int Pending = 0xFF00;
        public const int InvalidQueryRoot = 0xC001;

    }
    enum CtrlType
    {
        CTRL_C_EVENT = 0,
        CTRL_BREAK_EVENT = 1,
        CTRL_CLOSE_EVENT = 2,
        CTRL_LOGOFF_EVENT = 5,
        CTRL_SHUTDOWN_EVENT = 6
    }
}
