﻿using DicomObjects;
using DicomObjects.Enums;
using DicomObjects.UIDs;
using DicomObjects.Validation;
using Microsoft.Extensions.Logging;
using System;
using System.IO;
using System.Linq;

namespace KeyObjectNote.Core
{
    class Program
    {
        private const string ImagesPath = @"..\..\..\Images";
        private const string SelectObject_Message = "Select one or more objects first";
        private const string ValueType_Container = "CONTAINER";
        private const string ValueType_ContinuityOfContent = "SEPARATE";
        private const string ErrorMessage = "ERROR - more than one patient ID";
        private const string Path_Text = "PATH";
        private const string Hospital = "Hospital_";
        private const string PartOfUID = "1.2.3.4.5.6.7.8.100.";
        private const string PartOfRetrieveUrl = @"https:\\192.168.0.1\";
        private const string Modality = "KO";
        private const string CodeValue = "113030";
        private const string CodeSchemeDesignator = "DCM";
        private const string CodeMeaning = "Manifest";
        private const string FileName = "KOS.dcm";
        private const string PressAnyKeyToLoadImages = "Press any key to load images";
        private const string PressAnyKeyToGenerateKoObject = "Press any key to generate KO object";
        private const string PressAnyKeyToExit = "Press any key to exit";

        static void Main(string[] args)
        {
            ILoggerFactory loggerFactory = LoggerFactory.Create(config => config.AddConsole());
            ILogger logger = loggerFactory.CreateLogger<Program>();
            DicomGlobal.LogToLogger(logger, 0x7); // basic logging, change to 0x3F for more detailed logging
           
            Console.WriteLine(PressAnyKeyToLoadImages);
            Console.ReadKey();

            DicomDataSetCollection referencedInstances = new DicomDataSetCollection();
            DirectoryInfo di = new DirectoryInfo(ImagesPath);
            foreach (var img in di.GetFiles("*.*", SearchOption.TopDirectoryOnly))
            {
                if (DicomGlobal.IsDICOM(img.FullName))
                {
                    referencedInstances.Read(img.FullName);
                }
            }
            Console.WriteLine($"{referencedInstances.Count} images loaded");
            Console.WriteLine(PressAnyKeyToGenerateKoObject);
            Console.ReadKey();

            GenerateKO(referencedInstances);

            Console.WriteLine(PressAnyKeyToExit);
            Console.ReadKey();
        }

        private static void GenerateKO(DicomDataSetCollection referencedInstances)
        {
            DicomGlobal.LogToFile("C:\\DicomLogs", 0x3F);
            DicomDataSet KON = new DicomDataSet();
            DicomDataSet FirstObject;
            if (referencedInstances.Count == 0)
            {
                Console.WriteLine(SelectObject_Message);
                return;
            }
            FirstObject = referencedInstances[0];

            // General DICOM Information
            KON.SOPClass = SOPClasses.KeyObjectSelectionDocument;
            KON.StudyUID = FirstObject.StudyUID;
            KON.SeriesUID = DicomGlobal.NewUID();
            KON.InstanceUID = DicomGlobal.NewUID();
            KON.Add(Keyword.SeriesNumber, FirstObject.Value(Keyword.SeriesNumber));

            // Module: Patient Module (M)  - all taken from FirstObject
            KON.Add(Keyword.PatientName, FirstObject.Value(Keyword.PatientName));
            KON.Add(Keyword.PatientID, FirstObject.Value(Keyword.PatientID));
            KON.Add(Keyword.PatientBirthDate, FirstObject.Value(Keyword.PatientBirthDate));
            KON.Add(Keyword.PatientSex, FirstObject.Value(Keyword.PatientSex));

            // Module: General Study Module (M) - all taken from FirstObject
            KON.Add(Keyword.StudyDate, FirstObject.Value(Keyword.StudyDate));
            KON.Add(Keyword.StudyTime, FirstObject.Value(Keyword.StudyTime));
            KON.Add(Keyword.AccessionNumber, FirstObject.Value(Keyword.AccessionNumber));
            KON.Add(Keyword.ReferringPhysicianName, FirstObject.Value(Keyword.ReferringPhysicianName));
            KON.Add(Keyword.StudyID, FirstObject.Value(Keyword.StudyID));

            // Module: General Equipment Module (M)
            KON.Add(Keyword.Manufacturer, null);
            KON.Add(Keyword.SeriesNumber, 1);

            // Module: SR Document Content Module (M)
            KON.Add(Keyword.ValueType, ValueType_Container);
            KON.Add(Keyword.ContinuityOfContent, ValueType_ContinuityOfContent);
            {
                // SEQ -(0040,A043) - Concept Name Code Sequence
                DicomDataSetCollection ddsc = new DicomDataSetCollection();
                DicomDataSet dds = new DicomDataSet();
                dds.Add(0x0008, 0x0100, CodeValue);                   // Code Value
                dds.Add(0x0008, 0x0102, CodeSchemeDesignator);                      // Coding Scheme Designator
                dds.Add(0x0008, 0x0104, CodeMeaning);                 // Code Meaning
                ddsc.Add(dds);
                KON.Add(Keyword.ConceptNameCodeSequence, ddsc);
            }

            // Module: SOP Common Module (M)
            KON.Add(Keyword.InstanceNumber, 123456);

            // Module: Key Object Document Module (M) 
            KON.Add(Keyword.ContentDate, DateTime.Now.Date);
            KON.Add(Keyword.ContentTime, DateTime.Now.Date);

            DicomDataSet PseudoDicomDir = new DicomDataSet();  // Reference to the objects in the KON is in a hierarchy equivalent to a DicomDir (which DicomObjects can generate automatically)
            // hierarchy Study - Series - Instance
            foreach (DicomDataSet ThisObject in referencedInstances)
            {   
                PseudoDicomDir.AddToDirectory(ThisObject, Path_Text, ThisObject.OriginalTS);          // Add all the Objects to the DicomDir
            }

            if (PseudoDicomDir.Children.Count > 1)
                Console.WriteLine(ErrorMessage);        // The Key Object Note is limited to one Patient - Top level of the Dicom Dir is Patients

            DicomDataSetCollection StudySequence = new DicomDataSetCollection();// create a Study Level Sequence
            foreach (DicomDataSet study in PseudoDicomDir.Children[0].Children) // foreach Study in the dicomdir (only one patient allowed)
            {
                DicomDataSet StudySequenceItem = new DicomDataSet();
                StudySequenceItem.StudyUID = study.StudyUID;                    // Referenced Study Instance UID
                DicomDataSetCollection SeriesSequence = new DicomDataSetCollection();
                foreach (DicomDataSet series in study.Children)                 // for each series in each study
                {
                    int seriesCount = 1;
                    DicomDataSet SeriesSequenceItem = new DicomDataSet();
                    SeriesSequenceItem.SeriesUID = series.SeriesUID;            // Referenced Series Instance UID
                    DicomDataSetCollection InstanceSequence = new DicomDataSetCollection();
                    foreach (DicomDataSet Instance in series.Children)          // for each instance in each series
                    {
                        DicomDataSet InstanceSequenceItem = new DicomDataSet();
                        InstanceSequenceItem.Add(Keyword.ReferencedSOPClassUID, Instance.Value(Keyword.ReferencedSOPClassUIDInFile));
                        InstanceSequenceItem.Add(Keyword.ReferencedSOPInstanceUID, Instance.Value(Keyword.ReferencedSOPInstanceUIDInFile));
                        InstanceSequence.Add(InstanceSequenceItem);
                    }
                    SeriesSequenceItem.Add(Keyword.ReferencedSOPSequence, InstanceSequence); // add all the instance collection to the series
                    string aet = Hospital + seriesCount.ToString();
                    string uid = PartOfUID + seriesCount.ToString();
                    SeriesSequenceItem.Add(Keyword.RetrieveAETitle, aet);
                    SeriesSequenceItem.Add(Keyword.RetrieveLocationUID, uid);
                    SeriesSequenceItem.ValidationOptions = ValidationOptions.None;
                    SeriesSequenceItem.Add(Keyword.RetrieveURL, PartOfRetrieveUrl + aet + "\\" + series.SeriesUID);
                    SeriesSequence.Add(SeriesSequenceItem);
                }
                StudySequenceItem.Add(Keyword.ReferencedSeriesSequence, SeriesSequence);  // add the series collection to the study
                StudySequence.Add(StudySequenceItem);
            }
            KON.Add(Keyword.CurrentRequestedProcedureEvidenceSequence, StudySequence);  // Current Requested Procedure Evidence Sequence

            // Add Referenced Request Sequence
            DicomDataSetCollection refReqSQ = new DicomDataSetCollection();
            DicomDataSet refReqSQItem = new DicomDataSet();
            refReqSQItem.StudyUID = StudySequence[0].StudyUID;
            refReqSQItem.Add(Keyword.ReferencedStudySequence, StudySequence);
            refReqSQItem.Add(Keyword.AccessionNumber, FirstObject.Value(Keyword.AccessionNumber));
            refReqSQItem.Add(Keyword.PlacerOrderNumberImagingServiceRequest, "");
            refReqSQItem.Add(Keyword.FillerOrderNumberImagingServiceRequest, "");
            refReqSQItem.Add(Keyword.RequestedProcedureID, "");
            refReqSQItem.Add(Keyword.RequestedProcedureDescription, "");
            refReqSQItem.Add(Keyword.RequestedProcedureCodeSequence, new DicomDataSetCollection());
            refReqSQItem.Add(Keyword.RequestedProcedureID, "");
            refReqSQ.Add(refReqSQItem);
            KON.Add(Keyword.ReferencedRequestSequence, refReqSQ);
            // Module: Key Object Document Series Module (M)
            KON.Add(Keyword.Modality, Modality);
            KON.Add(Keyword.ReferencedPerformedProcedureStepSequence, FirstObject.Value(0x0008, 0x1111));
            KON.Add(Keyword.SeriesDate, DateTime.Now.Date);
            KON.Add(Keyword.SeriesTime, DateTime.Now.Date);

            KON.Write(FileName);
            Console.WriteLine(FileName+ " is written");
        }
    }
}
