﻿using DicomObjects;
using DicomObjects.DicomWeb;
using DicomObjects.Enums;
using DicomObjects.UIDs;
using Microsoft.Extensions.Logging;
using Serilog.Core;
using System;
using System.Net.Http;
using System.Net.Security;
using System.Security.Cryptography.X509Certificates;

namespace TestClient2
{
    class Program
    {
        protected Program() { }

        const bool asyncServerMode = true;
        static string serverURL;
        static ILogger logger;
        static void Main()
        {            
            if(asyncServerMode)
                serverURL = "https://localhost:9091";
            else
                serverURL = "https://localhost:9090";
            
            ILoggerFactory loggerFactory = LoggerFactory.Create(config => config.AddConsole());
            logger = loggerFactory.CreateLogger<Program>();
            LoggerHelper.SetLogger(logger);

            DicomDataSet ds = new("1.dcm");
            ds.InstanceUID = DicomGlobal.NewUID(); // create new InstanceUID every time to avoid duplicate on public dicomserver

            DicomGlobal.LogToLogger(logger, 0x3F);
            Console.WriteLine("Press any key to start Stow");
            Console.ReadKey();
            Console.WriteLine();
            Stow(ds);

            Console.WriteLine("Press any key to start Qido");
            Console.ReadKey();
            Console.WriteLine();
            Qido(ds);

            Console.WriteLine("Press any key to start Wado");
            Console.ReadKey();
            Console.WriteLine();
            Wado(ds);

            LoggerHelper.LogText(" ============= END OF TEST ============= ");
            Console.ReadKey();            
        }
        private static void Stow(DicomDataSet ds)
        {
            var dwc = new StowWebClient($"{serverURL}/Stow")
            {
                ServerCertificateCustomValidationCallback = ValidateWindowsCertificate
            };
            var dss = new DicomDataSetCollection() { ds };
            
            var r = dwc.Store(dss, TransferSyntaxes.ExplicitVRLittleEndian);
            LoggerHelper.LogText($"STOW response - {r.StatusCode}");
        }

        public static bool ValidateWindowsCertificate(HttpRequestMessage msg, X509Certificate2 certificate, X509Chain chain, SslPolicyErrors sslPolicyErrors)
        {
            // check sslPolicyErrors to validate server certificate errors
            // or simply return true to fully trust server certificate
            return true;
        }

        private static void Qido(DicomDataSet ds)
        {
            var dwc = new QidoWebClient($"{serverURL}/Qido")
            {
                ServerCertificateCustomValidationCallback = ValidateWindowsCertificate
            };
            var q = new DicomQuery
            {
                Level = QueryLevel.STUDY,
                Root = QueryRoot.Study
            };

            DicomDataSet queryDataSet = q.QueryDataSet();
            queryDataSet.Add(Keyword.PatientID, ds.PatientID);

            var result = dwc.Query(queryDataSet);
            Utils.Dump("QIDO", result);
        }      

        private static void Wado(DicomDataSet ds)
        {
            string studyUID = ds.StudyUID;
            string seriesUID = ds.SeriesUID;
            string instanceUID = ds.InstanceUID;

            var dwc = new WadoWebClient($"{serverURL}/Wado")
            {
                ServerCertificateCustomValidationCallback = ValidateWindowsCertificate
            };

            // retrieve native DICOM
            var result = dwc.RetrieveNative(studyUID, seriesUID, instanceUID);
            Utils.Dump("WADO Retrieve Native", result);

            // retrive native DICOM with only metadata (no blob)
            result = dwc.RetrieveMetaData(studyUID, seriesUID, instanceUID);
            Utils.Dump("WADO Retrieve MetaData", result);

            // retrieve rendered (image/jpeg, text/xml etc)
            var results = dwc.RetrieveRendered(studyUID, seriesUID, instanceUID);
            logger.LogInformation($"=============WADO Retrieve Rendered=============");
            logger.LogInformation($"Returned {results.Count} results:");
            foreach (var r in results)
            {
                logger.LogInformation($"    MediaType={r.Key.ContentType.MediaType}");
            }
        }
    }
}
